/**
 * Copyright since 2007 PrestaShop SA and Contributors
 * PrestaShop is an International Registered Trademark & Property of PrestaShop SA
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.md.
 * It is also available through the world-wide-web at this URL:
 * https://opensource.org/licenses/OSL-3.0
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade PrestaShop to newer
 * versions in the future. If you wish to customize PrestaShop for your
 * needs please refer to https://devdocs.prestashop.com/ for more information.
 *
 * @author    PrestaShop SA and Contributors <contact@prestashop.com>
 * @copyright Since 2007 PrestaShop SA and Contributors
 * @license   https://opensource.org/licenses/OSL-3.0 Open Software License (OSL 3.0)
 */

import EntitySearchInput from '@components/entity-search-input';

const {$} = window;

/**
 * This component is used in product page to selected where the redirection points to when the
 * product is out of stock. It is composed on two inputs:
 * - a selection of the redirection type
 * - a rich component to select a product or a category
 *
 * When the type is changed the component automatically updates the labels, remote search urls
 * and values of the target.
 */
export default class RedirectOptionManager {
  constructor($redirectTypeInput, $redirectTargetInput) {
    this.$redirectTypeInput = $redirectTypeInput;
    this.$redirectTargetInput = $redirectTargetInput;
    this.$redirectTargetRow = this.$redirectTargetInput.closest('.form-group');
    this.$redirectTargetLabel = $('.form-control-label', this.$redirectTargetRow).first();
    this.$redirectTargetHint = $('.typeahead-hint', this.$redirectTargetRow);
    this.buildAutoCompleteSearchInput();
    this.watchRedirectType();
  }

  /**
   * Watch the selected redirection type and adapt the inputs accordingly.
   */
  watchRedirectType() {
    this.lastSelectedType = this.$redirectTypeInput.val();

    this.$redirectTypeInput.change(() => {
      const redirectType = this.$redirectTypeInput.val();

      switch (redirectType) {
        case '301-category':
        case '302-category':
          this.entitySearchInput.setRemoteUrl(this.$redirectTargetInput.data('categorySearchUrl'));
          this.$redirectTargetInput.prop('placeholder', this.$redirectTargetInput.data('categoryPlaceholder'));
          this.$redirectTargetLabel.html(this.$redirectTargetInput.data('categoryLabel'));
          // If previous type was not a category we reset the selected value
          if (this.lastSelectedType !== '301-category' && this.lastSelectedType !== '302-category') {
            this.entitySearchInput.setValue(null);
          }
          this.$redirectTargetHint.html(this.$redirectTargetInput.data('categoryHelp'));
          this.showTarget();
          break;
        case '301-product':
        case '302-product':
          this.entitySearchInput.setRemoteUrl(this.$redirectTargetInput.data('productSearchUrl'));
          this.$redirectTargetInput.prop('placeholder', this.$redirectTargetInput.data('productPlaceholder'));
          this.$redirectTargetLabel.html(this.$redirectTargetInput.data('productLabel'));
          // If previous type was not a category we reset the selected value
          if (this.lastSelectedType !== '301-product' && this.lastSelectedType !== '302-product') {
            this.entitySearchInput.setValue(null);
          }
          this.$redirectTargetHint.html(this.$redirectTargetInput.data('productHelp'));
          this.showTarget();
          break;
        case '404':
        default:
          this.entitySearchInput.setValue(null);
          this.hideTarget();
          break;
      }
      this.lastSelectedType = this.$redirectTypeInput.val();
    });
  }

  buildAutoCompleteSearchInput() {
    this.entitySearchInput = new EntitySearchInput(this.$redirectTargetInput);
  }

  showTarget() {
    this.$redirectTargetRow.removeClass('d-none');
  }

  hideTarget() {
    this.$redirectTargetRow.addClass('d-none');
  }
}
