<?php

if (!function_exists('humani_display_comments')) {
    /**
     * Humani display comments
     *
     * @since  1.0.0
     */
    function humani_display_comments() {
        // If comments are open or we have at least one comment, load up the comment template.
        if (comments_open() || 0 !== intval(get_comments_number())) :
            comments_template();
        endif;
    }
}

if (!function_exists('humani_comment')) {
    /**
     * Humani comment template
     *
     * @param array $comment the comment array.
     * @param array $args the comment args.
     * @param int $depth the comment depth.
     *
     * @since 1.0.0
     */
    function humani_comment($comment, $args, $depth) {
        if ('div' === $args['style']) {
            $tag       = 'div';
            $add_below = 'comment';
        } else {
            $tag       = 'li';
            $add_below = 'div-comment';
        }
        ?>
        <<?php echo esc_attr($tag) . ' '; ?><?php comment_class(empty($args['has_children']) ? '' : 'parent'); ?> id="comment-<?php comment_ID(); ?>">
        <div class="comment-body">
        <div class="comment-meta commentmetadata">
            <div class="comment-author vcard">
                <?php echo get_avatar($comment, 128); ?>
                <?php printf('<cite class="fn">%s</cite>', get_comment_author_link()); ?>
            </div>
            <?php if ('0' === $comment->comment_approved) : ?>
                <em class="comment-awaiting-moderation"><?php esc_attr_e('Your comment is awaiting moderation.', 'humani'); ?></em>
                <br/>
            <?php endif; ?>

            <a href="<?php echo esc_url(htmlspecialchars(get_comment_link($comment->comment_ID))); ?>"
               class="comment-date">
                <?php echo '<time datetime="' . get_comment_date('c') . '">' . get_comment_date() . '</time>'; ?>
            </a>
        </div>
        <?php if ('div' !== $args['style']) : ?>
        <div id="div-comment-<?php comment_ID(); ?>" class="comment-content">
    <?php endif; ?>
        <div class="comment-text">
            <?php comment_text(); ?>
        </div>
        <div class="reply">
            <?php
            comment_reply_link(
                array_merge(
                    $args, array(
                        'add_below' => $add_below,
                        'depth'     => $depth,
                        'max_depth' => $args['max_depth'],
                    )
                )
            );
            ?>
            <?php edit_comment_link(esc_html__('Edit', 'humani'), '  ', ''); ?>
        </div>
        </div>
        <?php if ('div' !== $args['style']) : ?>
            </div>
        <?php endif; ?>
        <?php
    }
}

if (!function_exists('humani_credit')) {
    /**
     * Display the theme credit
     *
     * @return void
     * @since  1.0.0
     */
    function humani_credit() {
        ?>
        <div class="site-info">
            <?php echo apply_filters('humani_copyright_text', $content = esc_html__('Coppyright', 'humani') . ' &copy; ' . date('Y') . ' ' . '<a class="site-url" href="' . esc_url(site_url()) . '">' . esc_html(get_bloginfo('name')) . '</a>' . esc_html__('. All Rights Reserved.', 'humani')); ?>
        </div><!-- .site-info -->
        <?php
    }
}

if (!function_exists('humani_social')) {
    function humani_social() {
        $social_list = humani_get_theme_option('social_text', []);
        if (empty($social_list)) {
            return;
        }
        ?>
        <div class="humani-social">
            <ul>
                <?php

                foreach ($social_list as $social_item) {
                    ?>
                    <li><a href="<?php echo esc_url($social_item); ?>"></a></li>
                    <?php
                }
                ?>

            </ul>
        </div>
        <?php
    }
}

if (!function_exists('humani_site_branding')) {
    /**
     * Site branding wrapper and display
     *
     * @return void
     * @since  1.0.0
     */
    function humani_site_branding() {
        ?>
        <div class="site-branding">
            <?php echo humani_site_title_or_logo(); ?>
        </div>
        <?php
    }
}

if (!function_exists('humani_site_title_or_logo')) {
    /**
     * Display the site title or logo
     *
     * @param bool $echo Echo the string or return it.
     *
     * @return string
     * @since 2.1.0
     */
    function humani_site_title_or_logo() {
        ob_start();
        the_custom_logo(); ?>
        <div class="site-branding-text">
            <?php if (is_front_page()) : ?>
                <h1 class="site-title"><a href="<?php echo esc_url(home_url('/')); ?>"
                                          rel="home"><?php bloginfo('name'); ?></a></h1>
            <?php else : ?>
                <p class="site-title"><a href="<?php echo esc_url(home_url('/')); ?>"
                                         rel="home"><?php bloginfo('name'); ?></a></p>
            <?php endif; ?>

            <?php
            $description = get_bloginfo('description', 'display');

            if ($description || is_customize_preview()) :
                ?>
                <p class="site-description"><?php echo esc_html($description); ?></p>
            <?php endif; ?>
        </div><!-- .site-branding-text -->
        <?php
        $html = ob_get_clean();
        return $html;
    }
}

if (!function_exists('humani_primary_navigation')) {
    /**
     * Display Primary Navigation
     *
     * @return void
     * @since  1.0.0
     */
    function humani_primary_navigation() {
        ?>
        <nav class="main-navigation" role="navigation"
             aria-label="<?php esc_html_e('Primary Navigation', 'humani'); ?>">
            <?php
            $args = apply_filters('humani_nav_menu_args', [
                'fallback_cb'     => '__return_empty_string',
                'theme_location'  => 'primary',
                'container_class' => 'primary-navigation',
            ]);
            wp_nav_menu($args);
            ?>
        </nav>
        <?php
    }
}

if (!function_exists('humani_mobile_navigation')) {
    /**
     * Display Handheld Navigation
     *
     * @return void
     * @since  1.0.0
     */
    function humani_mobile_navigation() {
        ?>
        <div class="mobile-nav-tabs">
            <ul>
                <?php if (isset(get_nav_menu_locations()['handheld'])) { ?>
                    <li class="mobile-tab-title mobile-pages-title">
                        <i class="humani-icon-bars"></i>
                        <span><?php echo esc_html(get_term(get_nav_menu_locations()['handheld'], 'nav_menu')->name); ?></span>
                    </li>
                <?php } ?>
            </ul>
        </div>
        <nav class="mobile-menu-tab mobile-navigation mobile-pages-menu"
             aria-label="<?php esc_html_e('Mobile Navigation', 'humani'); ?>">
            <?php
            wp_nav_menu(
                array(
                    'theme_location'  => 'handheld',
                    'container_class' => 'handheld-navigation',
                )
            );
            ?>
        </nav>
        <?php
    }
}

if (!function_exists('humani_vertical_navigation')) {
    /**
     * Display Vertical Navigation
     *
     * @return void
     * @since  1.0.0
     */
    function humani_vertical_navigation() {

        if (isset(get_nav_menu_locations()['vertical'])) {
            $string = get_term(get_nav_menu_locations()['vertical'], 'nav_menu')->name;
            ?>
            <nav class="vertical-navigation" aria-label="<?php esc_html_e('Vertiacl Navigation', 'humani'); ?>">
                <div class="vertical-navigation-header">
                    <i class="humani-icon-caret-vertiacl-menu"></i>
                    <span class="vertical-navigation-title"><?php echo esc_html($string); ?></span>
                </div>
                <?php

                $args = apply_filters('humani_nav_menu_args', [
                    'fallback_cb'     => '__return_empty_string',
                    'theme_location'  => 'vertical',
                    'container_class' => 'vertical-menu',
                ]);

                wp_nav_menu($args);
                ?>
            </nav>
            <?php
        }
    }
}

if (!function_exists('humani_homepage_header')) {
    /**
     * Display the page header without the featured image
     *
     * @since 1.0.0
     */
    function humani_homepage_header() {
        edit_post_link(esc_html__('Edit this section', 'humani'), '', '', '', 'button humani-hero__button-edit');
        ?>
        <header class="entry-header">
            <?php
            the_title('<h1 class="entry-title">', '</h1>');
            ?>
        </header><!-- .entry-header -->
        <?php
    }
}

if (!function_exists('humani_page_header')) {
    /**
     * Display the page header
     *
     * @since 1.0.0
     */
    function humani_page_header() {

        if (is_front_page() || !is_page_template('default')) {
            return;
        }

        if (humani_is_elementor_activated()) {
            if (Humani_breadcrumb::get_template_id() !== '') {
                return;
            }
        }

        ?>
        <header class="entry-header">
            <?php
            if (has_post_thumbnail()) {
                humani_post_thumbnail('full');
            }
            the_title('<h1 class="entry-title">', '</h1>');
            ?>
        </header><!-- .entry-header -->
        <?php
    }
}

if (!function_exists('humani_page_content')) {
    /**
     * Display the post content
     *
     * @since 1.0.0
     */
    function humani_page_content() {
        ?>
        <div class="entry-content">
            <?php the_content(); ?>
            <?php
            wp_link_pages(
                array(
                    'before' => '<div class="page-links">' . esc_html__('Pages:', 'humani'),
                    'after'  => '</div>',
                )
            );
            ?>
        </div><!-- .entry-content -->
        <?php
    }
}

if (!function_exists('humani_post_header')) {
    /**
     * Display the post header with a link to the single post
     *
     * @since 1.0.0
     */
    function humani_post_header() {
        ?>
        <header class="entry-header">
            <?php

            /**
             * Functions hooked in to humani_post_header_before action.
             */
            do_action('humani_post_header_before');
            ?>

            <?php
            if (is_single()) {
                the_title('<h1 class="alpha entry-title">', '</h1>');
            } else {
                the_title(sprintf('<h2 class="alpha entry-title"><a href="%s" rel="bookmark">', esc_url(get_permalink())), '</a></h2>');
            }
            ?>

            <?php
            do_action('humani_post_header_after');
            ?>
        </header><!-- .entry-header -->
        <?php
    }
}

if (!function_exists('humani_post_content')) {
    /**
     * Display the post content with a link to the single post
     *
     * @since 1.0.0
     */
    function humani_post_content() {
        ?>
        <div class="entry-content">
            <?php

            /**
             * Functions hooked in to humani_post_content_before action.
             *
             */
            do_action('humani_post_content_before');


            if (is_single()) {

                the_content(
                    sprintf(
                    /* translators: %s: post title */
                        esc_html__('Read More', 'humani') . ' %s',
                        '<span class="screen-reader-text">' . get_the_title() . '</span>'
                    )
                );
            } else {
                the_excerpt();
            }

            /**
             * Functions hooked in to humani_post_content_after action.
             *
             */
            do_action('humani_post_content_after');

            wp_link_pages(
                array(
                    'before' => '<div class="page-links">' . esc_html__('Pages:', 'humani'),
                    'after'  => '</div>',
                )
            );
            ?>
        </div><!-- .entry-content -->
        <?php
    }
}

if (!function_exists('humani_post_meta')) {
    /**
     * Display the post meta
     *
     * @since 1.0.0
     */
    function humani_post_meta() {
        if ('post' !== get_post_type()) {
            return;
        }
        echo '<div class="entry-meta">';
        if (!has_post_thumbnail() || is_single()) {
            humani_categories_link();
        }
        // Posted on.
        $time_string = '<time class="entry-date published" datetime="%1$s">%2$s</time>';

        if (get_the_time('U') !== get_the_modified_time('U')) {
            $time_string = '<time class="entry-date published" datetime="%1$s">%2$s</time><time class="updated" datetime="%3$s">%4$s</time>';
        }

        $time_string = sprintf(
            $time_string,
            esc_attr(get_the_date('c')),
            esc_html(get_the_date()),
            esc_attr(get_the_modified_date('c')),
            esc_html(get_the_modified_date())
        );

        $posted_on = '<span class="posted-on">' . sprintf('<a href="%1$s" rel="bookmark">%2$s</a>', esc_url(get_permalink()), $time_string) . '</span>';

        // Author.
        $author = sprintf(
            '<span class="post-author"><span>%1$s<a href="%2$s" class="url fn" rel="author">%3$s</a></span></span>',
            esc_html__('By ', 'humani'),
            esc_url(get_author_posts_url(get_the_author_meta('ID'))),
            esc_html(get_the_author())
        );


        echo wp_kses(
            sprintf('%1$s %2$s', $posted_on, $author), array(
                'span' => array(
                    'class' => array(),
                ),
                'a'    => array(
                    'href'  => array(),
                    'title' => array(),
                    'rel'   => array(),
                ),
                'time' => array(
                    'datetime' => array(),
                    'class'    => array(),
                ),
            )
        );
        if (comments_open()) : ?>
            <span class="meta-comment">
                    <a class="comment-link" href="<?php echo esc_url(get_comments_link()); ?>"><i class="humani-icon-comments"></i> <?php echo ' ' . number_format_i18n(get_comments_number()); ?></a>
                </span>
        <?php endif;
        echo '</div>';
    }
}

if (!function_exists('humani_get_allowed_html')) {
    function humani_get_allowed_html() {
        return apply_filters(
            'humani_allowed_html',
            array(
                'br'     => array(),
                'i'      => array(),
                'b'      => array(),
                'u'      => array(),
                'em'     => array(),
                'del'    => array(),
                'a'      => array(
                    'href'  => true,
                    'class' => true,
                    'title' => true,
                    'rel'   => true,
                ),
                'strong' => array(),
                'span'   => array(
                    'style' => true,
                    'class' => true,
                ),
            )
        );
    }
}

if (!function_exists('humani_edit_post_link')) {
    /**
     * Display the edit link
     *
     * @since 2.5.0
     */
    function humani_edit_post_link() {
        edit_post_link(
            sprintf(
                wp_kses(__('Edit <span class="screen-reader-text">%s</span>', 'humani'),
                    array(
                        'span' => array(
                            'class' => array(),
                        ),
                    )
                ),
                get_the_title()
            ),
            '
        <div class="edit-link">',
            '
        </div>'
        );
    }
}

if (!function_exists('humani_categories_link')) {
    /**
     * Prints HTML with meta information for the current cateogries
     */
    function humani_categories_link() {

        // Get Categories for posts.
        $categories_list = get_the_category_list(' ');

        if ('post' === get_post_type() && $categories_list) {
            // Make sure there's more than one category before displaying.
            echo '
        <span class="categories-link"><span class="screen-reader-text">' . esc_html__('Categories', 'humani') . '</span>' . $categories_list . '</span>';
        }
    }
}

if (!function_exists('humani_post_taxonomy')) {
    /**
     * Display the post taxonomies
     *
     * @since 2.4.0
     */
    function humani_post_taxonomy() {
        /* translators: used between list items, there is a space after the comma */

        /* translators: used between list items, there is a space after the comma */
        $tags_list = get_the_tag_list('');
        ?>
        <aside class="entry-taxonomy">
            <?php if ($tags_list) : ?>
                <div class="tags-links">
                    <strong class="screen-reader-text"><?php echo esc_html(_n('Tag:', 'Tags:', count(get_the_tags()), 'humani')); ?></strong>
                    <?php printf('%s', $tags_list); ?>
                </div>
            <?php endif;
            humani_social_share();
            ?>
        </aside>
        <?php
    }
}

if (!function_exists('humani_paging_nav')) {
    /**
     * Display navigation to next/previous set of posts when applicable.
     */
    function humani_paging_nav() {
        global $wp_query;

        $args = array(
            'type'      => 'list',
            'next_text' => _x('NEXT', 'Next post', 'humani') . '<i class="humani-icon humani-icon-chevron-right"></i>',
            'prev_text' => '<i class="humani-icon humani-icon-chevron-left"></i>' . _x('PREVIOUS', 'Previous post', 'humani'),
        );

        the_posts_pagination($args);
    }
}

if (!function_exists('humani_post_nav')) {
    /**
     * Display navigation to next/previous post when applicable.
     */
    function humani_post_nav() {

        $prev_post = get_previous_post();
        $next_post = get_next_post();
        $args      = [];
        if ($next_post) {
            $args['next_text'] = '<span class="nav-content"><span class="reader-text">' . esc_html__('NEXT', 'humani') . ' </span><span class="title">%title</span></span> ' . get_the_post_thumbnail($next_post->ID, array(110, 110));
        }
        if ($prev_post) {
            $args['prev_text'] = get_the_post_thumbnail($prev_post->ID, array(110, 110)) . '<span class="nav-content"><span class="reader-text">' . esc_html__('PREV', 'humani') . ' </span><span class="title">%title</span></span> ';
        }

        the_post_navigation($args);

    }
}

if (!function_exists('humani_posted_on')) {
    /**
     * Prints HTML with meta information for the current post-date/time and author.
     *
     * @deprecated 2.4.0
     */
    function humani_posted_on() {
        _deprecated_function('humani_posted_on', '2.4.0');
    }
}

if (!function_exists('humani_homepage_content')) {
    /**
     * Display homepage content
     * Hooked into the `homepage` action in the homepage template
     *
     * @return  void
     * @since  1.0.0
     */
    function humani_homepage_content() {
        while (have_posts()) {
            the_post();

            get_template_part('content', 'homepage');

        } // end of the loop.
    }
}

if (!function_exists('humani_social_icons')) {
    /**
     * Display social icons
     * If the subscribe and connect plugin is active, display the icons.
     *
     * @link http://wordpress.org/plugins/subscribe-and-connect/
     * @since 1.0.0
     */
    function humani_social_icons() {
        if (class_exists('Subscribe_And_Connect')) {
            echo '<div class="subscribe-and-connect-connect">';
            subscribe_and_connect_connect();
            echo '</div>';
        }
    }
}

if (!function_exists('humani_get_sidebar')) {
    /**
     * Display humani sidebar
     *
     * @uses get_sidebar()
     * @since 1.0.0
     */
    function humani_get_sidebar() {
        get_sidebar();
    }
}

if (!function_exists('humani_post_thumbnail')) {
    /**
     * Display post thumbnail
     *
     * @param string $size the post thumbnail size.
     *
     * @uses has_post_thumbnail()
     * @uses the_post_thumbnail
     * @var $size thumbnail size. thumbnail|medium|large|full|$custom
     * @since 1.5.0
     */
    function humani_post_thumbnail($size = 'post-thumbnail') {
        echo '<div class="post-thumbnail">';
        if (has_post_thumbnail()) {
            the_post_thumbnail($size ? $size : 'post-thumbnail');
            if (!is_single()) {
                humani_categories_link();
            }
        }

        echo '</div>';
    }
}

if (!function_exists('humani_primary_navigation_wrapper')) {
    /**
     * The primary navigation wrapper
     */
    function humani_primary_navigation_wrapper() {
        echo '<div class="humani-primary-navigation"><div class="col-full">';
    }
}

if (!function_exists('humani_primary_navigation_wrapper_close')) {
    /**
     * The primary navigation wrapper close
     */
    function humani_primary_navigation_wrapper_close() {
        echo '</div></div>';
    }
}

if (!function_exists('humani_header_container')) {
    /**
     * The header container
     */
    function humani_header_container() {
        echo '<div class="col-full">';
    }
}

if (!function_exists('humani_header_container_close')) {
    /**
     * The header container close
     */
    function humani_header_container_close() {
        echo '</div>';
    }
}

if (!function_exists('humani_header_custom_link')) {
    function humani_header_custom_link() {
        echo humani_get_theme_option('custom-link', '');
    }

}

if (!function_exists('humani_header_contact_info')) {
    function humani_header_contact_info() {
        echo humani_get_theme_option('contact-info', '');
    }

}

if (!function_exists('humani_header_account')) {
    function humani_header_account() {

        ?>
        <div class="site-header-account">
            <a href="<?php echo esc_html(wp_login_url()); ?>">
                <i class="humani-icon-user-2"></i>
            </a>
            <div class="account-dropdown">

            </div>
        </div>
        <?php
    }

}

if (!function_exists('humani_template_account_dropdown')) {
    function humani_template_account_dropdown() {
        if (!humani_get_theme_option('show_header_account', true)) {
            return;
        }
        ?>
        <div class="account-wrap d-none">
            <div class="account-inner <?php if (is_user_logged_in()): echo "dashboard"; endif; ?>">
                <?php if (!is_user_logged_in()) {
                    humani_form_login();
                } else {
                    humani_account_dropdown();
                }
                ?>
            </div>
        </div>
        <?php
    }
}

if (!function_exists('humani_form_login')) {
    function humani_form_login() {
        ?>
        <div class="login-form-head">
            <span class="login-form-title"><?php esc_attr_e('Sign in', 'humani') ?></span>
            <span class="pull-right">
                <a class="register-link" href="<?php echo esc_url(wp_registration_url()); ?>"
                   title="<?php esc_attr_e('Register', 'humani'); ?>"><?php esc_attr_e('Create an Account', 'humani'); ?></a>
            </span>
        </div>
        <form class="humani-login-form-ajax" data-toggle="validator">
            <p>
                <label><?php esc_attr_e('Username or email', 'humani'); ?> <span class="required">*</span></label>
                <input name="username" type="text" required placeholder="<?php esc_attr_e('Username', 'humani') ?>">
            </p>
            <p>
                <label><?php esc_attr_e('Password', 'humani'); ?> <span class="required">*</span></label>
                <input name="password" type="password" required
                       placeholder="<?php esc_attr_e('Password', 'humani') ?>">
            </p>
            <button type="submit" data-button-action
                    class="btn btn-primary btn-block w-100 mt-1"><?php esc_html_e('Login', 'humani') ?></button>
            <input type="hidden" name="action" value="humani_login">
            <?php wp_nonce_field('ajax-humani-login-nonce', 'security-login'); ?>
        </form>
        <div class="login-form-bottom">
            <a href="<?php echo wp_lostpassword_url(get_permalink()); ?>" class="lostpass-link"
               title="<?php esc_attr_e('Lost your password?', 'humani'); ?>"><?php esc_attr_e('Lost your password?', 'humani'); ?></a>
        </div>
        <?php
    }
}

if (!function_exists('')) {
    function humani_account_dropdown() { ?>
        <?php if (has_nav_menu('my-account')) : ?>
            <nav class="social-navigation" role="navigation" aria-label="<?php esc_attr_e('Dashboard', 'humani'); ?>">
                <?php
                wp_nav_menu(array(
                    'theme_location' => 'my-account',
                    'menu_class'     => 'account-links-menu',
                    'depth'          => 1,
                ));
                ?>
            </nav><!-- .social-navigation -->
        <?php else: ?>
            <ul class="account-dashboard">
                <li>
                    <a href="<?php echo esc_url(get_dashboard_url(get_current_user_id())); ?>"
                       title="<?php esc_html_e('Dashboard', 'humani'); ?>"><?php esc_html_e('Dashboard', 'humani'); ?></a>
                </li>
                <li>
                    <a title="<?php esc_html_e('Log out', 'humani'); ?>" class="tips"
                       href="<?php echo esc_url(wp_logout_url(home_url())); ?>"><?php esc_html_e('Log Out', 'humani'); ?></a>
                </li>
            </ul>
        <?php endif;

    }
}

if (!function_exists('humani_header_search_popup')) {
    function humani_header_search_popup() {
        ?>
        <div class="site-search-popup">
            <div class="site-search-popup-wrap">
                <div class="site-search">
                    <?php get_search_form(); ?>
                </div>
                <a href="#" class="site-search-popup-close">
                    <svg class="close-icon" xmlns="http://www.w3.org/2000/svg" width="23.691" height="22.723" viewBox="0 0 23.691 22.723"><g transform="translate(-126.154 -143.139)"><line x2="23" y2="22" transform="translate(126.5 143.5)" fill="none" stroke="CurrentColor" stroke-width="1"></line><path d="M0,22,23,0" transform="translate(126.5 143.5)" fill="none" stroke="CurrentColor" stroke-width="1"></path></g></svg>
                </a>
            </div>
        </div>
        <?php
    }
}

if (!function_exists('humani_header_search_button')) {
    function humani_header_search_button() {
        ?>
        <div class="site-header-search">
            <a href="#" class="button-search-popup"><i class="humani-icon-search"></i></a>
            <?php  humani_header_search_popup(); ?>
        </div>
        <?php
    }
}


if (!function_exists('humani_header_sticky')) {
    function humani_header_sticky() {
        get_template_part('template-parts/header', 'sticky');
    }
}

if (!function_exists('humani_mobile_nav')) {
    function humani_mobile_nav() {
        if (isset(get_nav_menu_locations()['handheld']) || isset(get_nav_menu_locations()['vertical'])) {
            ?>
            <div class="humani-mobile-nav">
                <?php if (humani_is_elementor_activated()) { ?>
                    <div class="site-search">
                        <?php get_search_form(); ?>
                    </div>
                <?php } ?>
                <div class="menu-scroll-mobile">
                    <a href="#" class="mobile-nav-close"><i class="humani-icon-times"></i></a>
                    <?php
                    humani_mobile_navigation();
                    humani_social();
                    ?>
                </div>

            </div>
            <div class="humani-overlay"></div>
            <?php
        }
    }
}

if (!function_exists('humani_mobile_nav_button')) {
    function humani_mobile_nav_button() {
        if (isset(get_nav_menu_locations()['handheld'])) {
            ?>
            <a href="#" class="menu-mobile-nav-button">
				<span
                        class="toggle-text screen-reader-text"><?php echo esc_attr(apply_filters('humani_menu_toggle_text', esc_html__('Menu', 'humani'))); ?></span>
                <i class="humani-icon-bars"></i>
            </a>
            <?php
        }
    }
}

if (!function_exists('humani_footer_default')) {
    function humani_footer_default() {
        get_template_part('template-parts/copyright');
    }
}


if (!function_exists('humani_pingback_header')) {
    /**
     * Add a pingback url auto-discovery header for single posts, pages, or attachments.
     */
    function humani_pingback_header() {
        if (is_singular() && pings_open()) {
            echo '<link rel="pingback" href="', esc_url(get_bloginfo('pingback_url')), '">';
        }
    }
}


if (!function_exists('humani_social_share')) {
    function humani_social_share() {
        get_template_part('template-parts/socials');
    }
}

if (!function_exists('humani_update_comment_fields')) {
    function humani_update_comment_fields($fields) {

        $commenter = wp_get_current_commenter();
        $req       = get_option('require_name_email');
        $aria_req  = $req ? "aria-required='true'" : '';

        $fields['author']
            = '<p class="comment-form-author">
			<input id="author" name="author" type="text" placeholder="' . esc_attr__("Your Name *", "humani") . '" value="' . esc_attr($commenter['comment_author']) .
              '" size="30" ' . $aria_req . ' />
		</p>';

        $fields['email']
            = '<p class="comment-form-email">
			<input id="email" name="email" type="email" placeholder="' . esc_attr__("Email Address *", "humani") . '" value="' . esc_attr($commenter['comment_author_email']) .
              '" size="30" ' . $aria_req . ' />
		</p>';

        $fields['url']
            = '<p class="comment-form-url">
			<input id="url" name="url" type="url"  placeholder="' . esc_attr__("Your Website", "humani") . '" value="' . esc_attr($commenter['comment_author_url']) .
              '" size="30" />
			</p>';

        return $fields;
    }
}

add_filter('comment_form_default_fields', 'humani_update_comment_fields');

add_filter('bcn_breadcrumb_title', 'humani_breadcrumb_title_swapper', 3, 10);
function humani_breadcrumb_title_swapper($title, $type, $id) {
    if (in_array('home', $type)) {
        $title = esc_html__('Home', 'humani');
    }
    return $title;
}

function blog_humani_body_class($classes) {
    $sidebar   = humani_get_theme_option('blog_sidebar');
    $classes[] = '';
    if (is_active_sidebar('sidebar-blog')) {
        if ($sidebar == 'left') {
            $classes[] = 'humani-sidebar-left';
        }
    }
    return $classes;
}

add_filter('body_class', 'blog_humani_body_class', 1, 10);

function humani_replace_categories_list($output, $args) {
    if ($args['show_count'] = 1) {
        $pattern     = '#<li([^>]*)><a([^>]*)>(.*?)<\/a>\s*\(([0-9]*)\)\s*#i';  // removed ( and )
        $replacement = '<li$1><a$2><span class="cat-name">$3</span> <span class="cat-count">$4</span></a>';
        return preg_replace($pattern, $replacement, $output);
    }
    return $output;
}

add_filter('wp_list_categories', 'humani_replace_categories_list', 10, 2);

function humani_replace_archive_list($link_html, $url, $text, $format, $before, $after, $selected) {
    if ($format == 'html') {
        $pattern     = '#<li><a([^>]*)>(.*?)<\/a>&nbsp;\s*\(([0-9]*)\)\s*#i';  // removed ( and )
        $replacement = '<li><a$1><span class="archive-name">$2</span> <span class="archive-count">$3</span></a>';
        return preg_replace($pattern, $replacement, $link_html);
    }
    return $link_html;
}

add_filter('get_archives_link', 'humani_replace_archive_list', 10, 7);




