<?php

namespace GiveRecurring\Webhooks\Stripe\Listeners;

use Give\Framework\PaymentGateways\Log\PaymentGatewayLog;
use Give\PaymentGateways\Gateways\Stripe\Webhooks\StripeEventListener;
use Give\Subscriptions\Models\Subscription;
use Give\Subscriptions\ValueObjects\SubscriptionStatus;
use Stripe\Event;
use Stripe\Subscription as StripeSubscription;

/**
 * Class CustomerSubscriptionUpdated
 * @package GiveRecurring\Webhooks\Stripe\Listeners
 *
 * @since 2.10.0
 */
class CustomerSubscriptionUpdated extends StripeEventListener
{

    /**
     * @since 2.11.2 return early if subscription is not found
     * @since 2.10.0
     */
    public function processEvent(Event $event): void
    {
        /**
         * @since 2.10.0
         */
        do_action('give_recurring_stripe_processing_customer_subscription_updated', $event);

        /* @var StripeSubscription $stripeSubscription */
        $stripeSubscription = $event->data->object;
        $subscription = $this->getSubscription($event);

        if (!$subscription || !$this->isSubscriptionProcessable($stripeSubscription, $subscription)) {
            return;
        }

        $subscription->status = SubscriptionStatus::ACTIVE();
        $subscription->save();
    }

    /**
     * @inerhitDoc
     *
     * @since 2.10.0
     */
    protected function getFormId(Event $event): int
    {
        return $this->getSubscription($event)->donationFormId;
    }

    /**
     * @since 2.11.2 update return type to be nullable
     * @since 2.10.0
     */
    private function getSubscription(Event $event): ?Subscription
    {
        /* @var StripeSubscription $stripeSubscription */
        $stripeSubscription = $event->data->object;

        $subscription = give()->subscriptions->queryByGatewaySubscriptionId($stripeSubscription->id)->get();

        if (!$subscription){
            PaymentGatewayLog::http('CustomerSubscriptionUpdated: Subscription not found', ['subscriptionId' => $stripeSubscription->id]);

            return null;
        }

        if (!$subscription->id && $this->hasDonationIdInInvoiceMetadata($stripeSubscription)) {
            $donationMetaData = $stripeSubscription->metadata->toArray();
            $donationId = !empty($donationMetaData['Donation Post ID']) ? $donationMetaData['Donation Post ID'] : 0;
            $subscription = give()->subscriptions->queryByDonationId($donationId)->get();
        }

        return $subscription;
    }

    /**
     * @since 2.10.0
     */
    private function isSubscriptionProcessable(StripeSubscription $stripeSubscription, Subscription $subscription): bool
    {
        return $subscription->id && $subscription->status === SubscriptionStatus::PAUSED() && !$stripeSubscription->pause_collection;
    }

    /**
     * @since 2.10.0
     */
    private function hasDonationIdInInvoiceMetadata(StripeSubscription $stripeSubscription): bool
    {
        return $stripeSubscription->metadata &&
            array_key_exists('Donation Post ID', $stripeSubscription->metadata->toArray());
    }
}
