<?php
namespace GiveRecurring\PaymentGateways\PayPalCommerce\Repositories;

use Give\PaymentGateways\PayPalCommerce\PayPalClient;
use Give_Subscription;
use GiveRecurring\Infrastructure\Log;
use GiveRecurring\PaymentGateways\PayPalCommerce\HttpHeader;
use GiveRecurring\PaymentGateways\PayPalCommerce\Models\Subscriber;
use GiveRecurring\PaymentGateways\PayPalCommerce\Models\Plan as PlanModel;
use GiveRecurring\PaymentGateways\PayPalCommerce\PayPalHttpRequests\ActivateSubscription;
use GiveRecurring\PaymentGateways\PayPalCommerce\PayPalHttpRequests\CreateSubscription;
use GiveRecurring\PaymentGateways\PayPalCommerce\PayPalHttpRequests\GetSubscription;
use GiveRecurring\PaymentGateways\PayPalCommerce\PayPalHttpRequests\UpdateSubscriptionStatus;
use Exception;
use PayPalHttp\HttpException;
use PayPalHttp\IOException;
use RuntimeException;

/**
 * Class Subscription
 * @package GiveRecurring\PaymentGateways\PayPalCommerce\Repositories
 *
 * @since 1.11.0
 */
class Subscription{
	/**
	 * @var PayPalClient
	 *
	 * @since 1.11.0
	 */
	private $payPalClient;

	/**
	 * Product constructor.
	 *
	 * @param  PayPalClient  $payPalClient
	 *
	 * @since 1.11.0
	 */
	public function __construct( PayPalClient $payPalClient ){
		$this->payPalClient = $payPalClient;
	}

    /**
     * Create PayPal Subscription
     *
     * @since 2.6.0 Remove unnecessary try catch block. Implement PalPal Client and add function argument type.
     * @sicne 1.11.0
     *
     * @throws HttpException
     * @throws IOException
     */
    public function create(PlanModel $plan, Subscriber $subscriber, array $redirectUrls = []): array
    {
        $payPalRequestBody = $this->getCreateSubscriptionQuery($plan, $subscriber, $redirectUrls);
        $payPalResponse = $this->payPalClient->getHttpClient()->execute(
            new CreateSubscription(
                give(HttpHeader::class)->getHeaders(
                    ['PayPal-Partner-Attribution-Id' => give('PAYPAL_COMMERCE_SUBSCRIPTION_ATTRIBUTION_ID')]
                ),
                $payPalRequestBody
            )
        );

        if (
            $payPalResponse->statusCode !== 201
            || ! property_exists($payPalResponse->result, 'id')
        ) {
            Log::error(
                'PayPal Commerce Create Subscription Api Request Failure',
                [
                    'response' => $payPalResponse,
                    'requestBody' => $payPalRequestBody,
                ]
            );

            throw new RuntimeException(
                esc_html__(
                    'Sorry, We are failed to create PayPal Subscription. Please try after some time',
                    'give-recurring'
                )
            );
        }

        return json_decode(json_encode($payPalResponse->result), true);
    }

    /**
     * Create PayPal Subscription
     *
     * @since 2.6.0 Remove unnecessary try catch block. Implement PalPal Client.
     * @sicne 1.11.0
     * @throws Exception
     */
    public function getFromPayPal(string $paypalSubscriptionId): array
    {
        $payPalResponse = $this->payPalClient->getHttpClient()->execute(
            new GetSubscription(
                give(HttpHeader::class)->getHeaders(),
                $paypalSubscriptionId
            )
        );

        if (
            $payPalResponse->statusCode !== 200
            || ! property_exists($payPalResponse->result, 'id')
        ) {
            Log::error(
                'PayPal Commerce Get Subscription Api Request Failure',
                [
                    'response' => $payPalResponse,
                    'subscriptionId' => $paypalSubscriptionId
                ]
            );

            throw new RuntimeException(
                esc_html__(
                    'Sorry, We are failed to create PayPal Subscription. Please try after some time',
                    'give-recurring'
                )
            );
        }

        return json_decode(json_encode($payPalResponse->result), true);
    }

	/**
	 * Return query to create subscription in array format.
	 *
	 * @since 1.11.0
	 *
	 * @param  PlanModel  $plan
	 * @param  Subscriber  $subscriber
	 * @param array $redirectUrls
	 *
	 * @return array
	 */
	private function getCreateSubscriptionQuery( PlanModel $plan, Subscriber $subscriber, $redirectUrls ){
		$query = [
			'plan_id'        => $plan->id,
			'subscriber' => [
				'name' => [
					'given_name' => $subscriber->firstName,
					'surname' => $subscriber->lastName
				],
				'email_address' => $subscriber->emailAddress
			],
			'application_context' => [
				'locale' => str_replace( '_', '-', get_locale() ),
				'shipping_preference' => 'NO_SHIPPING',
				'user_action' => 'SUBSCRIBE_NOW',
			]
		];

		if( array_key_exists( 'return', $redirectUrls ) ) {
			$query['application_context']['return_url'] = $redirectUrls['return'];
		}

		if( array_key_exists( 'cancel', $redirectUrls ) ) {
			$query['application_context']['cancel_url'] = $redirectUrls['cancel'];
		}

		if ( $subscriber->hasCard() ) {
			// @see https://developer.paypal.com/docs/api/subscriptions/v1/#definition-payment_source.card
			$query['subscriber']['payment_source']['card'] = [
				'number'        => $subscriber->card->number,
				'name'          => $subscriber->card->name,
				'expiry'        => $subscriber->card->expiry,
				'security_code' => $subscriber->card->cvc,
			];

			unset( $query['application_context'] );
		}

		return $query;
	}

	/**
	 * Get subscription bu paypal subscription id.
	 *
	 * @since 1.11.0
	 *
	 * @param string $id
	 *
	 * @return Give_Subscription
	 */
	public static function getSubscriptionByPayPalId( $id ){
		return new Give_Subscription( $id, true );
	}

    /**
     * Update donation status.
     *
     * @since 2.6.0 Remove unnecessary try catch block. Implement PalPal Client.
     * @since 1.11.0
     *
     * @throws Exception
     */
    public function updateStatus(string $paypalSubscriptionId, string $status)
    {
        $payPalResponse = $this->payPalClient->getHttpClient()->execute(
            new UpdateSubscriptionStatus(
                give(HttpHeader::class)->getHeaders(),
                $paypalSubscriptionId,
                $status
            )
        );

        if ($payPalResponse->statusCode !== 204) {
            Log::error(
                'PayPal Commerce Update Subscription Status Api Request Failure',
                [
                    'response' => $payPalResponse,
                    'status' => $status,
                    'subscriptionId' => $paypalSubscriptionId
                ]
            );

            throw new RuntimeException(
                esc_html__(
                    'Sorry, We are update PayPal Subscription status. Please try after some time',
                    'give-recurring'
                )
            );
        }
    }

    /**
     * @since 2.12.0
     * @throws Exception
     */
    public function activate(string $paypalSubscriptionId): void
    {
        $payPalResponse = $this->payPalClient->getHttpClient()->execute(
            new ActivateSubscription(
                give(HttpHeader::class)->getHeaders(),
                $paypalSubscriptionId
            )
        );

        if ($payPalResponse->statusCode !== 204) {
            Log::error(
                'PayPal Commerce Update Subscription Status Api Request Failure',
                [
                    'response' => $payPalResponse,
                    'subscriptionId' => $paypalSubscriptionId
                ]
            );

            throw new RuntimeException(
                esc_html__(
                    'Sorry, We are update PayPal Subscription status. Please try after some time',
                    'give-recurring'
                )
            );
        }
    }
}
