<?php
namespace GiveRecurring\PaymentGateways\PayPalCommerce\Repositories;

use Exception;
use Give\PaymentGateways\PayPalCommerce\PayPalClient;
use GiveRecurring\Infrastructure\Log;
use GiveRecurring\PaymentGateways\PayPalCommerce\HttpHeader;
use GiveRecurring\PaymentGateways\PayPalCommerce\Models\Product as ProductModel;
use GiveRecurring\PaymentGateways\PayPalCommerce\PayPalHttpRequests\CreateProduct;
use GiveRecurring\PaymentGateways\PayPalCommerce\PayPalHttpRequests\GetProduct;
use RuntimeException;

/**
 * Class Product
 * @package GiveRecurring\PaymentGateways\PayPalCommerce\Repositories
 *
 * @since 1.11.0
 */
class Product{
	/**
	 * Form meta key for product details.
	 *
	 * @since 1.11.0
	 * @var string
	 */
	const PRODUCT_FORM_META_KEY = 'paypal_commerce_product';

	/**
	 * Product id prefix.
	 *
	 * @since 1.11.0
	 * @var string
	 */
	const PRODUCT_ID_PREFIX = 'GIVE-';

	/**
	 * @var PayPalClient
	 *
	 * @since 1.11.0
	 */
	private $payPalClient;

	/**
	 * Product constructor.
	 *
	 * @param  PayPalClient  $payPalClient
	 *
	 * @since 1.11.0
	 */
	public function __construct( PayPalClient $payPalClient ){
		$this->payPalClient = $payPalClient;
	}

	/**
	 * Get PayPal Product.
	 *
	 * @since 1.11.0
	 *
	 * @param  int  $formId
	 *
	 * @return ProductModel|null
	 */
	public function get( $formId ){
		$product = give()->form_meta->get_meta( $formId, ProductModel::getProductDetailFormMetaKey(), true );

		if( ! $product ) {
			return null;
		}

		return new ProductModel( $product['id'] );
	}

    /**
     * Create product.
     *
     * @see https://developer.paypal.com/docs/api/catalog-products/v1/#products_create
     *
     * @since 2.6.0 Remove unnecessary try catch block. Implement PalPal Client and type to function param.
     * @since 1.11.0
     *
     * @throws Exception
     */
    public function create(int $formId): array
    {
        $payPalRequestBody = [
            'id' => self::PRODUCT_ID_PREFIX . $formId,
            'name' => get_the_title($formId),
            'type' => 'SERVICE',
            'category' => 'CHARITY',
        ];

        try {
            $payPalResponse = $this->payPalClient->getHttpClient()->execute(
                new CreateProduct(
                    give(HttpHeader::class)->getHeaders(),
                    $payPalRequestBody
                )
            );

            if (
                $payPalResponse->statusCode !== 201
                || ! property_exists($payPalResponse->result, 'id')
            ) {
                Log::error(
                    'PayPal Commerce Create Product Api Request Failure',
                    [
                        'response' => $payPalResponse,
                        'requestBody' => $payPalRequestBody,
                    ]
                );

                throw new RuntimeException(
                    esc_html__(
                        'Sorry, We are failed to create PayPal product. Please try after some time',
                        'give-recurring'
                    )
                );
            }
        } catch (\Exception $exception) {
            $payPalResponseInArrayFormat = json_decode($exception->getMessage(), true);

            if ($this->isDuplicateProduct($payPalResponseInArrayFormat)) {
                return $this->getProductFromPayPal($formId);
            }

            throw $exception;
        }

        return json_decode(json_encode($payPalResponse->result), true);
    }

	/**
	 * Return whether or not produce exist n PayPal.
	 *
	 * @since 1.11.0
	 *
	 * @param  array  $payPalResponse
	 *
	 * @return bool
	 */
	private function isDuplicateProduct( $payPalResponse ){
		if( ! array_key_exists( 'details', $payPalResponse ) ){
			return false;
		}

		$status = current(
			array_filter(
				$payPalResponse['details'],
				static function ( $error ) {
					return 'DUPLICATE_RESOURCE_IDENTIFIER' === $error['issue'];
				}
			)
		);

		return (bool) $status;
	}

	/**
	 * Store product detail.
	 *
	 * @since 1.11.0
	 *
	 * @param int $formId
	 * @param string $metaKey
	 * @param array $product
	 *
	 * @return bool
	 */
	public function storeProductDetail( $formId, $metaKey, $product ){
		return give()->form_meta->update_meta( $formId, $metaKey, $product );
	}

	/**
	 * Get product details from PayPal.
     *
     * @see https://developer.paypal.com/docs/api/catalog-products/v1/#products_get
	 *
     * @since 2.6.0 Remove unnecessary try catch block. Implement PalPal Client and add function argument type.
	 * @since 1.11.0
	 *
	 * @throws Exception
	 */
    private function getProductFromPayPal( int $formId ): array
    {
        $payPalResponse = $this->payPalClient->getHttpClient()->execute(
            new GetProduct(
                give(HttpHeader::class)->getHeaders(),
                self::PRODUCT_ID_PREFIX . $formId
            )
        );

        if (
            $payPalResponse->statusCode !== 200
            || ! property_exists($payPalResponse->result, 'id')
        ) {
            Log::error(
                'PayPal Commerce Get Product Api Request Failure',
                [
                    'response' => $payPalResponse,
                ]
            );

            throw new RuntimeException(
                esc_html__(
                    'Sorry, We are failed to fetch PayPal product. Please try after some time',
                    'give-recurring'
                )
            );

        }

        return json_decode(json_encode($payPalResponse->result), true);
    }
}
