<?php

namespace GiveRecurring\PaymentGatewayModules\Modules\Razorpay;

use Exception;
use Give\Donations\Models\Donation;
use Give\Donations\Models\DonationNote;
use Give\Donations\ValueObjects\DonationStatus;
use Give\Framework\PaymentGateways\Commands\SubscriptionComplete;
use Give\Framework\PaymentGateways\Contracts\Subscription\SubscriptionDashboardLinkable;
use Give\Framework\PaymentGateways\Exceptions\PaymentGatewayException;
use Give\Framework\PaymentGateways\Log\PaymentGatewayLog;
use Give\Framework\PaymentGateways\SubscriptionModule;
use Give\Subscriptions\Models\Subscription;
use Give\Subscriptions\ValueObjects\SubscriptionStatus;
use Give_RazorPay_Subscriptions;
use GiveRazorpay\Actions\MaybeResetRazorpaySessionData;
use GiveRazorpay\DataTransferObjects\RazorpayGatewayData;

/**
 * @since 2.9.0
 */
class RazorpayGatewaySubscriptionModule extends SubscriptionModule implements SubscriptionDashboardLinkable
{
    /**
     * @since 2.9.0
     *
     * @throws Exception
     */
    public function createSubscription(
        Donation $donation,
        Subscription $subscription,
        $gatewayData
    ): SubscriptionComplete
    {
        try {
            /** @var  $razorpayGatewayData RazorpayGatewayData */
            $razorpayGatewayData = $gatewayData['razorpayGatewayData'];

            $this->validateRazorpaySubscription($subscription, $razorpayGatewayData);

            // Store Customer ID.
            give_razorpay_save_customer_id($donation->id, $razorpayGatewayData->razorpayCustomerId);

            DonationNote::create([
                'donationId' => $donation->id,
                'content' => sprintf(
                    __('Transaction Successful. Razorpay Transaction ID: %s | Razorpay Subscription ID: %s | Razorpay Customer ID: %s',
                        'give-razorpay'),
                    $razorpayGatewayData->razorpayPaymentId,
                    $razorpayGatewayData->razorpaySubscriptionId,
                    $razorpayGatewayData->razorpayCustomerId

                ),
            ]);

            give_update_meta($donation->id, 'razorpay_donation_response', $razorpayGatewayData->razorpayResponse);
            give_update_meta($donation->id, 'donor_phone', $razorpayGatewayData->razorpayDonorPhone);

            give(MaybeResetRazorpaySessionData::class)($donation->formId);

            return new SubscriptionComplete($razorpayGatewayData->razorpayPaymentId,
                $razorpayGatewayData->razorpaySubscriptionId);
        } catch (Exception $e) {
            $subscription->status = SubscriptionStatus::FAILING();
            $subscription->save();

            $donation->status = DonationStatus::FAILED();
            $donation->save();

            $errorMessage = $e->getMessage();

            DonationNote::create([
                'donationId' => $donation->id,
                'content' => sprintf(esc_html__('Donation failed. Reason: %s', 'give-recurring'), $errorMessage),
            ]);

            throw new PaymentGatewayException($errorMessage);
        }
    }

    /**
     * @since 2.9.0
     *
     * @throws Exception
     */
    public function cancelSubscription(Subscription $subscription): bool
    {
        try {
            $cancelled = $this->cancelRazorpaySubscription($subscription);

            if ($cancelled) {
                $subscription->status = SubscriptionStatus::CANCELLED();
                $subscription->save();
            }

            return $cancelled;
        } catch (Exception $e) {
            throw new PaymentGatewayException($e->getMessage());
        }
    }

    /**
     * @since 2.9.0
     */
    public function gatewayDashboardSubscriptionUrl(Subscription $subscription): string
    {
        $base = 'https://dashboard.razorpay.com/app/subscriptions/';

        return esc_url($base . $subscription->gatewaySubscriptionId);
    }

    /**
     * @since 2.9.0
     *
     * @throws Exception
     */
    protected function validateRazorpaySubscription(
        Subscription $subscription,
        RazorpayGatewayData $razorpayGatewayData
    ): bool {
        if (empty($razorpayGatewayData->razorpayCustomerId)) {
            throw new PaymentGatewayException(__('Missing Razorpay Customer ID', 'give-recurring'));
        }

        if (empty($razorpayGatewayData->razorpaySubscriptionId)) {
            throw new PaymentGatewayException(__('Missing Razorpay Subscription ID', 'give-recurring'));
        }

        if ( ! give_razorpay_validate_payment($subscription->donationFormId, $razorpayGatewayData->razorpayResponse)) {
            throw new PaymentGatewayException(__('Invalid Razorpay Subscription', 'give-recurring'));
        }

        if ( ! empty($razorpayGatewayData->razorpayResponse['error_code'])) {
            PaymentGatewayLog::error(
                sprintf('[Razorpay] The subscription creation failed. API response error: %s',
                    $razorpayGatewayData->razorpayResponse['error_code']),
                [
                    'Subscription ID' => $subscription->id,
                    'Donation ID' => $subscription->initialDonation()->id,
                    'Error Code' => $razorpayGatewayData->razorpayResponse['error_code'],
                    'Full Response' => $razorpayGatewayData->razorpayResponse,
                ]
            );

            throw new PaymentGatewayException(sprintf(__('The subscription creation failed. Details: %s',
                'give-recurring'),
                $razorpayGatewayData->razorpayResponse['error_code']));
        }

        return true;
    }

    /**
     * @since 2.9.0
     */
    protected function cancelRazorpaySubscription(Subscription $subscription): bool
    {
        $razorpay_subscription = new Give_RazorPay_Subscriptions();

        return $razorpay_subscription->cancel_subscription($subscription->gatewaySubscriptionId);
    }
}
