<?php

namespace GiveRecurring\PaymentGatewayModules\Modules;

use Exception;
use Give\Donations\Models\Donation;
use Give\Framework\PaymentGateways\SubscriptionModule;
use Give\Framework\PaymentGateways\Commands\SubscriptionProcessing;
use Give\Framework\Support\ValueObjects\Money;
use Give\Subscriptions\Models\Subscription;
use Give\Subscriptions\ValueObjects\SubscriptionStatus;

/**
 * @since 2.13.0
 */
class OfflineGatewaySubscriptionModule extends SubscriptionModule
{
    /**
     * @since 2.14.0 update to use SubscriptionProcessing command
     * @since 2.13.0
     */
    public function createSubscription(
        Donation $donation,
        Subscription $subscription,
        $gatewayData = null
    ): SubscriptionProcessing {
        // For offline donations, we use the donation ID as the transaction ID
        // and generate a simple subscription ID since there's no external gateway
        $gatewayTransactionId = "offline-transaction-id-$donation->id";
        $gatewaySubscriptionId = "offline-subscription-id-$subscription->id";

        return new SubscriptionProcessing(
            $gatewaySubscriptionId,
            $gatewayTransactionId,

        );
    }

    /**
     * @since 2.13.0
     *
     * @throws Exception
     */
    public function cancelSubscription(Subscription $subscription)
    {
        // For offline subscriptions, we simply update the status to cancelled
        // as there's no external gateway to communicate with
        $subscription->status = SubscriptionStatus::CANCELLED();
        $subscription->save();
    }

    /**
     * @since 2.13.0
     *
     * @throws Exception
     */
    public function updateSubscriptionAmount(Subscription $subscription, Money $newRenewalAmount)
    {
        // For offline subscriptions, we can directly update the amount
        // as there's no external gateway to synchronize with
        $subscription->amount = $newRenewalAmount;
        $subscription->save();
    }

    /**
     * @since 2.13.0
     *
     * @inheritDoc
     */
    public function canSyncSubscriptionWithPaymentGateway(): bool
    {
        // Offline subscriptions cannot be synchronized with an external gateway
        return false;
    }

    /**
     * @since 2.13.0
     *
     * @inheritDoc
     */
    public function canUpdateSubscriptionAmount(): bool
    {
        // Offline subscriptions can have their amount updated locally
        return true;
    }

    /**
     * @since 2.13.0
     *
     * @inheritDoc
     */
    public function canUpdateSubscriptionPaymentMethod(): bool
    {
        // Offline subscriptions don't have payment methods to update
        return false;
    }
}
