<?php

namespace GiveRecurring\PaymentGatewayModules\Modules\Mollie;

use Exception;
use Give\Donations\Models\Donation;
use Give\Donations\Models\DonationNote;
use Give\Donations\ValueObjects\DonationStatus;
use Give\Framework\Http\Response\Types\RedirectResponse;
use Give\Framework\PaymentGateways\Commands\RedirectOffsite;
use Give\Framework\PaymentGateways\Contracts\Subscription\SubscriptionDashboardLinkable;
use Give\Framework\PaymentGateways\Exceptions\PaymentGatewayException;
use Give\Framework\PaymentGateways\Log\PaymentGatewayLog;
use Give\Framework\PaymentGateways\SubscriptionModule;
use Give\Subscriptions\Models\Subscription;
use Give\Subscriptions\ValueObjects\SubscriptionStatus;
use GiveMollie\Actions\CreateMolliePayment;
use GiveMollie\API\MollieApi;
use GiveRecurring\PaymentGatewayModules\Modules\Mollie\Actions\CancelMollieSubscription;
use GiveRecurring\PaymentGatewayModules\Modules\Mollie\Actions\CreateMollieSubscription;
use Mollie\Api\Types\SequenceType;

/**
 * @since 2.7.0
 */
class MollieGatewaySubscriptionModule extends SubscriptionModule implements SubscriptionDashboardLinkable
{
    /**
     * @since 2.7.0
     */
    public $secureRouteMethods = [
        'handleSuccessSubscriptionReturn',
        'handleCanceledSubscriptionReturn',
    ];

    /**
     * In order to get started with recurring payments you need to require the customer’s consent through a first
     * payment. It’s similar to a regular payment, but the customer is shown information about your organization,
     * and the customer needs to complete the payment with the account or card that will be used for recurring
     * charges in the future. After the first payment is completed successfully, the customer’s account or card
     * will immediately be chargeable on-demand, or periodically through subscriptions.
     *
     * @see https://docs.mollie.com/payments/recurring#payments-recurring-first-payment
     *
     * @since 2.7.0
     *
     * @throws Exception
     */
    public function createSubscription(Donation $donation, Subscription $subscription, $gatewayData): RedirectOffsite
    {
        try {
            $paymentParameters = array_merge(
                $this->gateway->getPaymentParameters($donation, $gatewayData),
                ['sequenceType' => SequenceType::SEQUENCETYPE_FIRST] // A first payment for the customer to agree to automatic recurring charges.
            );
            $paymentParameters['redirectUrl'] = $this->getSubscriptionReturnURL($donation, $subscription, $gatewayData);
            $paymentParameters['cancelUrl'] = $this->getSubscriptionCancelURL($donation, $subscription, $gatewayData);
            $paymentParameters['description'] = MollieApi::getPaymentDescription($donation);

            $molliePayment = (new CreateMolliePayment())($donation, $paymentParameters);
            $donation->gatewayTransactionId = $molliePayment->id;
            $donation->save();

            return new RedirectOffsite($molliePayment->getCheckoutUrl());
        } catch (Exception $e) {
            $subscription->status = SubscriptionStatus::FAILING();
            $subscription->save();

            $donation->status = DonationStatus::FAILED();
            $donation->save();

            $errorMessage = $e->getMessage();

            DonationNote::create([
                'donationId' => $donation->id,
                'content' => sprintf(esc_html__('Donation failed. Reason: %s', 'give-mollie'), $errorMessage),
            ]);

            throw new PaymentGatewayException($errorMessage);
        }
    }

    /**
     * @since 2.7.0
     *
     * @throws Exception
     */
    protected function handleSuccessSubscriptionReturn(array $queryParams): RedirectResponse
    {
        $subscription = Subscription::find((int)$queryParams['subscription-id']);

        try {
            $mollieSubscription = (new CreateMollieSubscription())($subscription);

            $subscription->gatewaySubscriptionId = $mollieSubscription->id;
            $subscription->status = SubscriptionStatus::ACTIVE();
            $subscription->save();

            PaymentGatewayLog::success(
                sprintf('Subscription active. Subscription ID: %s', $subscription->id),
                [
                    'Payment Gateway' => $subscription->gateway()->getId(),
                    'Gateway Subscription Id' => $subscription->gatewaySubscriptionId,
                    'Gateway Transaction Id' => $subscription->initialDonation()->gatewayTransactionId,
                    'Subscription Id' => $subscription->id,
                    'Donation Id' => $subscription->initialDonation()->id,
                ]
            );
        } catch (Exception $e) {
            $subscription->status = SubscriptionStatus::FAILING();
            $subscription->save();

            PaymentGatewayLog::error(
                sprintf('[Mollie] Subscription not completed on Mollie side. Error:  %s',
                    $e->getCode() . ' - ' . $e->getMessage()),
                [
                    'Payment Gateway' => $subscription->gateway()->getId(),
                    'Gateway Subscription Id' => $subscription->gatewaySubscriptionId,
                    'Gateway Transaction Id' => $subscription->initialDonation()->gatewayTransactionId,
                    'Subscription Id' => $subscription->id,
                    'Donation Id' => $subscription->initialDonation()->id,
                ]
            );
        }

        return new RedirectResponse(esc_url_raw($queryParams['givewp-return-url']));
    }

    /**
     * @since 2.7.0
     * @throws Exception
     */
    protected function handleCanceledSubscriptionReturn(array $queryParams): RedirectResponse
    {
        $donation = Donation::find((int)$queryParams['donation-id']);
        $donation->status = DonationStatus::CANCELLED();
        $donation->save();

        $subscription = Subscription::find((int)$queryParams['subscription-id']);
        $subscription->status = SubscriptionStatus::CANCELLED();
        $subscription->save();


        return new RedirectResponse(esc_url_raw($queryParams['givewp-return-url']));
    }

    /**
     * @since 2.7.0
     */
    private function getSubscriptionReturnURL(Donation $donation, Subscription $subscription, $gatewayData): string
    {
        $successReturnUrl = $this->gateway->generateSecureGatewayRouteUrl(
            'handleSuccessSubscriptionReturn',
            $donation->id,
            [
                'donation-id' => $donation->id,
                'subscription-id' => $subscription->id,
                'givewp-return-url' => $gatewayData['successUrl'],
            ]
        );

        /**
         * Filter the success return URL.
         *
         * @since 2.7.0
         *
         * @param string $successReturnUrl The success return URL
         * @param int    $donationId       The donation ID
         */
        return apply_filters('givewp_mollie_success_return_url', $successReturnUrl, $donation->id);
    }

    /**
     * @since 2.7.0
     */
    private function getSubscriptionCancelURL(Donation $donation, Subscription $subscription, $gatewayData): string
    {
        $cancelReturnUrl = $this->gateway->generateSecureGatewayRouteUrl(
            'handleCanceledSubscriptionReturn',
            $donation->id,
            [
                'donation-id' => $donation->id,
                'subscription-id' => $subscription->id,
                'givewp-return-url' => $gatewayData['cancelUrl'],
            ]
        );

        /**
         * Filter the cancel return URL.
         *
         * @since 2.7.0
         *
         * @param string $cancelReturnUrl The cancel return URL
         * @param int    $donationId      The donation ID
         */
        return apply_filters('givewp_mollie_cancel_return_url', $cancelReturnUrl, $donation->id);
    }

    /**
     * @since 2.7.0
     *
     * @throws PaymentGatewayException
     */
    public function cancelSubscription(Subscription $subscription): bool
    {
        try {
            (new CancelMollieSubscription())($subscription);
            $subscription->status = SubscriptionStatus::CANCELLED();
            $subscription->save();
        } catch (Exception $e) {
            throw new PaymentGatewayException($e->getMessage());
        }

        return true;
    }

    /**
     * Mollie doesn't provide a direct link for subscriptions, so we are using the
     * customer link which lists all subscriptions associated with the customer.
     *
     * @since 2.7.0
     */
    public function gatewayDashboardSubscriptionUrl(Subscription $subscription): string
    {
        $mollieCustomerId = Give()->donor_meta->get_meta($subscription->donorId, MollieApi::getCustomerMetaKey(), true);

        return esc_url("https://www.mollie.com/dashboard/customers/$mollieCustomerId");
    }
}
