<?php

namespace GiveRecurring\PaymentGatewayModules\Modules\Mollie\Actions;

use Exception;
use Give\Framework\PaymentGateways\Exceptions\PaymentGatewayException;
use Give\Subscriptions\Models\Subscription;
use GiveMollie\Actions\GetMollieCustomer;
use GiveMollie\API\MollieApi;
use Mollie\Api\Resources\Subscription as MollieSubscription;

/**
 * @see https://docs.mollie.com/reference/v2/subscriptions-api/create-subscription
 *
 * @since 2.7.0
 */
class CreateMollieSubscription
{
    /**
     * @since 2.7.0
     * @throws PaymentGatewayException
     */
    public function __invoke(Subscription $subscription): MollieSubscription
    {
        try {
            $mollieCustomer = (new GetMollieCustomer())($subscription->initialDonation());

            return $mollieCustomer->createSubscription($this->getSubscriptionParameters($subscription));
        } catch (Exception $e) {
            throw new PaymentGatewayException(
                sprintf(
                    __('[Mollie] Failed to create subscription. API Error: %s', 'give-mollie'),
                    $e->getCode() . ' - ' . $e->getMessage()
                )
            );
        }
    }

    /**
     * @since 2.7.0
     */
    private function getSubscriptionParameters(Subscription $subscription): array
    {
        $params = [
            'amount' => [
                'value' => $subscription->initialDonation()->amount->formatToDecimal(),
                'currency' => $subscription->initialDonation()->amount->getCurrency()->getCode(),
            ],
            'startDate' => $subscription->renewsAt->format('Y-m-d'),
            'interval' => $this->getSubscriptionInterval($subscription),
            'webhookUrl' => $this->getSubscriptionWebhookUrl($subscription),
            'description' => MollieApi::getPaymentDescription($subscription->initialDonation()),
        ];

        //Subtract 1 because of the initial payment.
        if ($subscription->installments > 0) {
            $params['times'] = $subscription->installments - 1;
        }

        return $params;
    }

    /**
     * Possible values: ... months ... weeks ... days
     * Example: 1 month or 14 days
     * IMPORTANT: The maximum interval is 1 year (12 months, 52 weeks or 365 days).
     *
     * @see https://docs.mollie.com/reference/v2/subscriptions-api/create-subscription
     *
     * @since 2.7.0
     */
    private function getSubscriptionInterval(Subscription $subscription): string
    {
        $multiple = $subscription->frequency > 1;
        $mollieInterval = '1 month';

        if ($subscription->period->isDay()) {
            $mollieInterval = $subscription->frequency . ($multiple ? ' days' : ' day');
        } elseif ($subscription->period->isWeek()) {
            $mollieInterval = $subscription->frequency . ($multiple ? ' weeks' : ' week');
        } elseif ($subscription->period->isMonth()) {
            $mollieInterval = $subscription->frequency . ($multiple ? ' months' : ' month');
        } elseif ($subscription->period->isQuarter()) {
            $mollieInterval = $subscription->frequency * 3 . ' months';
        } elseif ($subscription->period->isYear()) {
            $mollieInterval = '12 months';
        }

        return $mollieInterval;
    }

    /**
     * @since 2.7.0
     */
    private function getSubscriptionWebhookUrl(Subscription $subscription): string
    {
        $webhookListener = $subscription->gateway()->getWebhookNotificationsListener();

        return $subscription->gateway()->generateGatewayRouteUrl($webhookListener, [
            'hook_type' => 'subscription',
            'give-sub-id' => $subscription->id,
        ]);
    }
}
