<?php

namespace GiveRecurring\PaymentGatewayModules\Modules\Blink\Actions;

use Give\Framework\Support\Facades\ActionScheduler\AsBackgroundJobs;
use GiveRecurring\PaymentGatewayModules\Modules\Blink\DataTransferObjects\BlinkSubscriptionWebhookRequestDTO;

/**
 * @since 2.11.0
 */
class HandleBlinkSubscriptionWebhookRequest
{
    /**
     * @since 2.11.0
     */
    public function __invoke(BlinkSubscriptionWebhookRequestDTO $blinkWebhookRequest)
    {
        if ($this->isOneTimeDonation($blinkWebhookRequest)) {
            return;
        }

        if (! $this->isRenewalNotification($blinkWebhookRequest)) {
            return;
        }

        $this->handlePaymentStatus($blinkWebhookRequest);
    }

    /**
     * @since 2.11.0
     */
    private function isOneTimeDonation(BlinkSubscriptionWebhookRequestDTO $blinkWebhookRequest): bool
    {
        return 'payments' === $blinkWebhookRequest->hookType;
    }

    /**
     * @since 2.11.0
     */
    public function isRenewalNotification(BlinkSubscriptionWebhookRequestDTO $blinkWebhookRequest): bool
    {
        return (
            ! empty($blinkWebhookRequest->scheduleId) &&
            ! empty($blinkWebhookRequest->recurringId) &&
            ! empty($blinkWebhookRequest->status)
        );
    }

    /**
     * Handle the payment status and enqueue the appropriate async action.
     *
     * @since 1.0.0
     */
    private function handlePaymentStatus(BlinkSubscriptionWebhookRequestDTO $blinkWebhookRequest): void
    {
        $status = strtolower($blinkWebhookRequest->status);

        switch ($status) {
            case 'tendered':
            case 'captured':
            case 'success':
            case 'accept':
                $this->setSubscriptionRenewalPaymentAsyncEvent(
                    $blinkWebhookRequest->scheduleId,
                    $blinkWebhookRequest->recurringId
                );

                return;
        }

        $this->setSubscriptionFailingAsyncEvent($blinkWebhookRequest->scheduleId);
    }

    /**
     * @since 2.11.0
     */
    private function setSubscriptionRenewalPaymentAsyncEvent(
        string $gatewaySubscriptionId,
        string $gatewayTransactionId
    ) {
        $this->enqueueAsyncAction(
            'givewp_blink_event_subscription_renewal_donation',
            [$gatewaySubscriptionId, $gatewayTransactionId]
        );
    }

    /**
     * @since 2.11.0
     */
    private function setSubscriptionFailingAsyncEvent(string $gatewaySubscriptionId)
    {
        $this->enqueueAsyncAction('givewp_blink_event_subscription_failing', [$gatewaySubscriptionId]);
    }

    /**
     * @since 2.11.0
     */
    private function enqueueAsyncAction(string $hook, array $args)
    {
        AsBackgroundJobs::enqueueAsyncAction($hook, $args, 'give-recurring-blink');
    }
}
