<?php

declare(strict_types=1);

namespace GiveFormFieldManager\FormFields\Rules;

use Closure;
use Give\Framework\Exceptions\Primitives\InvalidArgumentException;
use Give\VendorOverrides\Validation\Exceptions\ValidationException;
use Give\Vendors\StellarWP\Validation\Contracts\ValidatesOnFrontEnd;
use Give\Vendors\StellarWP\Validation\Contracts\ValidationRule;
class MaxRepeat implements ValidationRule, ValidatesOnFrontEnd
{
    /**
     * @var int
     */
    private $size;

    /**
     * @inheritDoc
     *
     * @since 2.1.0
     */
    public static function id(): string
    {
        return 'maxRepeat';
    }

    /**
     * @inheritDoc
     *
     * @since 2.1.0
     */
    public static function fromString(string $options = null): ValidationRule
    {
        if (!is_numeric($options)) {
            throw new InvalidArgumentException('Max Repeat validation rule requires a numeric value');
        }

        return new self((int)$options);
    }

    /**
     * @since 2.1.0
     */
    public function __construct(int $size)
    {
        if ($size <= 0) {
            throw new InvalidArgumentException('Max Repeat validation rule requires a non-negative value');
        }

        $this->size = $size;
    }

    /**
     * @inheritDoc
     *
     * @since 2.1.0
     *
     * @throws ValidationException
     */
    public function __invoke($value, Closure $fail, string $key, array $values)
    {
        if (is_int($value) || is_float($value)) {
            if ($value > $this->size) {
                $fail(sprintf(__('%s must be less than or equal to %d', 'give-form-field-manager'), '{field}', $this->size));
            }
        } elseif (is_string($value)) {
            if (mb_strlen($value) > $this->size) {
                $fail(sprintf(__('%s must be less than or equal to %d characters', 'give-form-field-manager'), '{field}', $this->size));
            }
        } else {
            throw new ValidationException("Field value must be a number or string");
        }
    }

    /**
     * @inheritDoc
     *
     * @since 2.1.0
     */
    public function serializeOption(): int
    {
        return $this->size;
    }

    /**
     * @since 2.1.0
     */
    public function getSize(): int
    {
        return $this->size;
    }

    /**
     * @since 2.1.0
     *
     * @return void
     */
    public function size(int $size)
    {
        if ($size <= 0) {
            throw new InvalidArgumentException('Max Repeat validation rule requires a non-negative value');
        }

        $this->size = $size;
    }
}
