<?php

namespace GiveFormFieldManager\FormFields\Fields;

use Give\Framework\FieldsAPI\Concerns;
use Give\Framework\FieldsAPI\Field;
use GiveFormFieldManager\FormFields\Fields\Contracts\HasFormInputValidator;
use GiveFormFieldManager\FormFields\Fields\FieldValidators\RepeaterFieldValidator;
use GiveFormFieldManager\FormFields\Rules\MaxRepeat;

/**
 * @since 2.0.0
 */
class Repeater extends Field implements HasFormInputValidator{
	use Concerns\HasEmailTag;
	use Concerns\HasHelpText;
	use Concerns\HasLabel;
	use Concerns\HasMaxLength;
	use Concerns\HasPlaceholder;

	const TYPE = 'ffm-repeat';

	/** @var string[] */
	protected $columns = [];

	/**
	 * Add columns to the repeater
	 *
	 * @since 2.0.0
	 *
 	 * @param string[] $columns
 	 * @return $this
 	 */
	public function columns( ...$columns ) {
		$this->columns = $columns;

		return $this;
	}

	/**
	 * Get the columns.
	 *
	 * @since 2.0.0
	 *
 	 * @return string[]
 	 */
	public function getColumns() {
		return $this->columns;
	}

	/**
	 * Set the maximum times the field/row can repeat.
	 *
     * @since 2.1.0 update to use new validation rule system
	 * @since 2.0.0
	 */
	public function maxRepeatable(int $maxRepeatable): self
    {
        if ( $this->hasRule('maxRepeat') ) {
            /** @var MaxRepeat $rule */
            $rule = $this->getRule('maxRepeat');
            $rule->size($maxRepeatable);
        } else {
            $this->rules("maxRepeat:$maxRepeatable");
        }

		return $this;
	}

	/**
	 * Get the maximum times the field/row can repeat.
	 *
     * @since 2.1.0 update to use new validation rule system
	 * @since 2.0.0
	 *
	 * @return int|null
	 */
	public function getMaxRepeatable() {
        if ( $this->hasRule('maxRepeat') ) {
            /** @var MaxRepeat $rule */
            $rule = $this->getRule('maxRepeat');
            return $rule->getSize();
        }

        return null;
	}

    /**
     * @since 2.0.3
     * @inerhitDoc
     * @return string
     */
    public function getFormInputValidator()
    {
        return RepeaterFieldValidator::class;
    }
}
