import {InspectorControls} from '@wordpress/block-editor';
import {Button, PanelBody, PanelRow, SelectControl, ToggleControl} from '@wordpress/components';
import {createInterpolateElement, useState} from '@wordpress/element';
import {__, sprintf} from '@wordpress/i18n';

import {ConditionalLogicAttribute, SelectControlOption} from '../types';
import LogicStatementsPopover from './LogicStatementsPopover';
import {conditionOptions, labels} from '../constants';

type Props = {
    conditionalLogic: ConditionalLogicAttribute;
    setAttributes: (newAttributes: object) => void;
    fieldOptions: SelectControlOption[];
};

/**
 * Renders the conditional logic inspector controls.
 *
 * @since 3.0.0
 */
export function Edit({conditionalLogic, setAttributes, fieldOptions}: Props) {
    const [isPopoverOpen, setIsPopoverOpen] = useState<boolean>(false);
    const AfterDisplaySettingsFill = window.givewp.form.slots.AfterDisplaySettingsFill;

    const openPopover = () => setIsPopoverOpen(true);
    const closePopover = () => setIsPopoverOpen(false);

    const setConditionalLogicAttributes = (newAttributes: Partial<ConditionalLogicAttribute>) => {
        setAttributes({
            conditionalLogic: {
                ...conditionalLogic,
                ...newAttributes,
            },
        });
    };

    const ConditionFieldHelpText = () => {
        return createInterpolateElement(
            sprintf(
                /* translators: 1: Action name (show/hide) 2: Match type (all/any) */
                __(
                    '<strong>%1$s</strong> this field if <strong>%2$s</strong> of the logic statement below matches',
                    'give'
                ),
                labels.actions[conditionalLogic.action],
                labels.booleans[conditionalLogic.boolean].toLowerCase()
            ),
            {
                strong: <strong />,
            }
        );
    };

    return (
        <>
            <InspectorControls>
                <AfterDisplaySettingsFill>
                    <PanelBody
                        title={__('Conditional logic', 'give')}
                        initialOpen={true}
                        className={'givewp-conditional-logic'}
                    >
                        <PanelRow>
                            <ToggleControl
                                label={__('Enable conditional logic', 'give')}
                                help={__(
                                    'Sets this field visibility based on the value of another field in this form.',
                                    'give'
                                )}
                                checked={conditionalLogic.enabled}
                                onChange={(value) => {
                                    setConditionalLogicAttributes({enabled: value});
                                }}
                            />
                        </PanelRow>
                        {conditionalLogic.enabled && (
                            <>
                                <PanelRow>
                                    <SelectControl
                                        label={__('Condition', 'give')}
                                        help={<ConditionFieldHelpText />}
                                        options={conditionOptions}
                                        value={`${conditionalLogic.action}-${conditionalLogic.boolean}`}
                                        onChange={(value: string) => {
                                            const [action, boolean] = value.split('-') as [
                                                ConditionalLogicAttribute['action'],
                                                ConditionalLogicAttribute['boolean']
                                            ];
                                            setConditionalLogicAttributes({action, boolean});
                                        }}
                                    />
                                </PanelRow>
                                <PanelRow>
                                    <Button
                                        variant="secondary"
                                        onClick={openPopover}
                                        className={'givewp-conditional-logic__open-popover-button'}
                                    >
                                        {__('Add logic statement', 'give')}
                                    </Button>
                                    <LogicStatementsPopover
                                        visible={isPopoverOpen}
                                        onClose={closePopover}
                                        setConditionalLogicAttributes={setConditionalLogicAttributes}
                                        conditionalLogic={conditionalLogic}
                                        fieldOptions={fieldOptions}
                                    />
                                </PanelRow>
                            </>
                        )}
                    </PanelBody>
                </AfterDisplaySettingsFill>
            </InspectorControls>
        </>
    );
}
