import {Button, SelectControl} from '@wordpress/components';
import {__} from '@wordpress/i18n';

import {labels} from '../../constants';
import {minusCircle, plusCircle} from '../../icons';
import ValueField from './ValueField';
import {ConditionalLogicAttribute, ConditionalLogicRule, SelectControlOption} from '../../types';

type Props = {
    conditionalLogic: ConditionalLogicAttribute;
    fieldOptions: SelectControlOption[];
    onAddRule: () => void;
    onUpdateRule: (index: number) => (key: keyof ConditionalLogicRule, value: string) => void;
    onRemoveRule: (index: number) => () => void;
};

export default function PopoverContent({conditionalLogic, fieldOptions, onAddRule, onUpdateRule, onRemoveRule}: Props) {
    return (
        <div className={'givewp-conditional-logic__popover-content'}>
            <div className={'givewp-conditional-logic__popover-content__add-rule'}>
                <p>{__('Logic statement', 'give')}</p>
                <Button icon={plusCircle} onClick={onAddRule} />
            </div>
            <div className={'givewp-conditional-logic__popover-content__rules'}>
                {conditionalLogic.rules.length > 0 ? (
                    conditionalLogic.rules.map((rule, index) => (
                        <div key={index} className={'givewp-conditional-logic__popover-content__rule'}>
                            <SelectControl
                                options={[
                                    {
                                        label: __('Select a field', 'give-form-field-manager'),
                                        value: '',
                                    },
                                    ...fieldOptions,
                                ]}
                                value={rule.field}
                                onChange={(newValue: string) => onUpdateRule(index)('field', newValue)}
                            />
                            <SelectControl
                                value={rule.operator}
                                onChange={(newValue: string) => onUpdateRule(index)('operator', newValue)}
                                options={Object.keys(labels.operators).map((operator) => ({
                                    label: labels.operators[operator],
                                    value: operator,
                                }))}
                            />
                            <ValueField
                                rule={rule}
                                handleChange={(newValue: string) => onUpdateRule(index)('value', newValue)}
                            />
                            <Button icon={minusCircle} onClick={onRemoveRule(index)} />
                        </div>
                    ))
                ) : (
                    <p className={'givewp-conditional-logic__popover-content__no-rule'}>
                        {__('No logic statement added yet.', 'give')}{' '}
                        <Button variant={'link'} onClick={onAddRule}>
                            {__('Add', 'give')}
                        </Button>
                    </p>
                )}
            </div>
        </div>
    );
}
