import {useMemo, useState} from '@wordpress/element';
import AceEditor from 'react-ace';
import {__} from '@wordpress/i18n';
import {BaseControl, Button, Modal, PanelRow} from '@wordpress/components';

import 'ace-builds/src-noconflict/mode-html';
import 'ace-builds/src-noconflict/snippets/html';
import 'ace-builds/src-noconflict/theme-textmate';
import 'ace-builds/src-noconflict/ext-language_tools';

export default function CodeEditor({value, onChange}) {
    const [isOpen, setOpen] = useState<boolean>(false);

    const openModal = () => setOpen(true);
    const closeModal = () => setOpen(false);

    const codeEditorControl = useMemo(() => {
        return (
            <AceEditor
                name={'html-code-editor'}
                mode="html"
                theme="textmate"
                onLoad={(editor) => {
                    editor.renderer.setScrollMargin(8, 8, 0, 0);
                    editor.renderer.setPadding(8);
                }}
                value={value}
                debounceChangePeriod={500}
                onChange={onChange}
                showPrintMargin={false}
                highlightActiveLine={false}
                showGutter={true}
                maxLines={isOpen ? Infinity : 12}
                wrapEnabled={true}
                minLines={isOpen ? 12 : 6}
                width="100%"
                setOptions={{
                    enableBasicAutocompletion: true,
                    enableLiveAutocompletion: true,
                    enableSnippets: true,
                    showLineNumbers: true,
                    tabSize: 4,
                    useWorker: false,
                }}
            />
        );
    }, [isOpen, onChange, value]);

    return (
        <PanelRow>
            <BaseControl label={__('HTML Code')}>
                <div
                    style={{
                        border: '1px solid #949494',
                        borderRadius: '2px',
                    }}
                >
                    {codeEditorControl}
                </div>
                <Button
                    variant={'secondary'}
                    onClick={openModal}
                    style={{width: '100%', marginTop: 'var(--givewp-spacing-2)', justifyContent: 'center'}}
                >
                    {__('Edit in Modal', 'give-form-field-manager')}
                </Button>
            </BaseControl>

            {!!isOpen && (
                <Modal
                    overlayClassName="components-modal__screen-overlay--givewpwp-custom-css"
                    title={__('HTML Code', 'give-form-field-manager')}
                    onRequestClose={closeModal}
                    style={{
                        height: '100%',
                        maxHeight: '100%', // Override the max height of the modal component.
                        width: '500px',
                        position: 'absolute',
                        right: '0',
                    }}
                >
                    <div
                        style={{
                            margin: '0 -31px -23px -31px', // Offset the modal padding in order to fill the available space.
                        }}
                    >
                        {codeEditorControl}
                    </div>
                </Modal>
            )}
        </PanelRow>
    );
}
