import {__} from '@wordpress/i18n';
import {InspectorControls} from '@wordpress/block-editor';
import {BaseControl, CheckboxControl, PanelBody, PanelRow, TextControl} from '@wordpress/components';

import metadata from './block.json';
import Icon from './Icon';
import PlaceholderControl from '../components/PlaceholderControl';
import './styles.scss';

type fileTypeOptionProps = {
    label: string;
    value: string;
    help?: string;
};

const fileTypes: fileTypeOptionProps[] = [
    {
        label: __('Images', 'give-form-field-manager'),
        value: 'image',
        help: '(jpg, jpeg, png, gif, bmp)',
    },
    {
        label: __('Audio', 'give-form-field-manager'),
        value: 'audio',
        help: '(mp3, wav, ogg, wma, mka, m4a, ra, mid, midi)',
    },
    {
        label: __('Video', 'give-form-field-manager'),
        value: 'video',
        help: '(avi, divx, flv, mov, ogv, mkv, mp4, m4v, divx, mpg, mpeg, mpe)',
    },
    {
        label: __('PDF', 'give-form-field-manager'),
        value: 'pdf',
    },
    {
        label: __('CSV', 'give-form-field-manager'),
        value: 'csv',
    },
    {
        label: __('Executable files (exe)', 'give-form-field-manager'),
        value: 'exe',
    },
    {
        label: __('Zip archives', 'give-form-field-manager'),
        value: 'zip',
        help: '(zip, gx, rar, gzip, 7z)',
    },
    {
        label: __('Office documents', 'give-form-field-manager'),
        value: 'office',
        help: '(doc, ppt, pps, xls, mdb, docx, xlsx, pptx, odt, odp, odg, odc, odb, odf, rtf, txt)',
    },
];

/**
 * @since 3.0.0
 */
const settings = {
    ...metadata,
    icon: Icon,
    edit: ({attributes, setAttributes}) => {
        const {label, description, isRequired, maxFileSize, allowedFileTypes} = attributes;
        const {AfterFieldSettingsFill} = window.givewp.form.slots;

        const handleAllowedFileTypesChange =
            (fileType: string) =>
            (checked: boolean): void => {
                const newAllowedFileTypes = checked
                    ? [...allowedFileTypes, fileType]
                    : allowedFileTypes.filter((type: string): boolean => type !== fileType);

                setAttributes({allowedFileTypes: newAllowedFileTypes});
            };

        return (
            <>
                <InspectorControls>
                    <AfterFieldSettingsFill>
                        <PanelBody title={__('File types', 'give-form-field-manager')}>
                            <PanelRow>
                                <TextControl
                                    label={__('Max File Size', 'give-form-field-manager')}
                                    type={'number'}
                                    value={maxFileSize}
                                    onChange={(maxFileSize) => setAttributes({maxFileSize})}
                                    help={__('Enter maximum upload size limit in KB', 'give-form-field-manager')}
                                />
                            </PanelRow>
                            <PanelRow>
                                <BaseControl label={'Upload File Types'}>
                                    {fileTypes.map((fileType: fileTypeOptionProps) => {
                                        const {label, value, help = ''} = fileType;

                                        return (
                                            <CheckboxControl
                                                label={label}
                                                checked={allowedFileTypes.includes(value)}
                                                onChange={handleAllowedFileTypesChange(value)}
                                                help={help}
                                                className={'give-ffm-fileupload__checkbox'}
                                            />
                                        );
                                    })}
                                </BaseControl>
                            </PanelRow>
                        </PanelBody>
                    </AfterFieldSettingsFill>
                </InspectorControls>
                <PlaceholderControl label={label} description={description} isRequired={isRequired}>
                    <input type="file" className={'give-ffm-fileupload__input'} readOnly />
                </PlaceholderControl>
            </>
        );
    },
};

const fileUpload = {
    name: settings.name,
    settings,
};

export default fileUpload;
