import {CheckboxControl, PanelRow, SelectControl, TextControl} from '@wordpress/components';
import {useState} from '@wordpress/element';
import {__} from '@wordpress/i18n';
import {InspectorControls} from '@wordpress/block-editor';
import {Markup} from 'interweave';
import ControlForPopover from '../components/ControlForPopover';
import PlaceholderControl from '../components/PlaceholderControl';
import StyledPopover from '../components/StyledPopover';

import metadata from './block.json';
import Icon from './Icon';

import './styles.scss';
import {ClassicEditor} from '@givewp/form-builder-library/';

/**
 * @since 3.0.0
 */
const settings = {
    ...metadata,
    icon: Icon,
    edit: ({attributes, setAttributes}) => {
        const {
            label,
            isRequired,
            checkboxLabel,
            displayType,
            linkText,
            linkUrl,
            agreementText,
            modalHeading,
            modalAcceptanceText,
        } = attributes;
        const {FieldSettingsFill} = window.givewp.form.slots;

        const [showAgreementTextModal, setShowAgreementTextModal] = useState(false);
        const isModalDisplay = displayType === 'showModalTerms';
        const isLinkDisplay = displayType === 'showLinkTerms';
        const isFormDisplay = displayType === 'showFormTerms';

        return (
            <>
                <PlaceholderControl label={label} isRequired={isRequired}>
                    <CheckboxPlaceholder
                        label={checkboxLabel}
                        linkText={linkText}
                        isFormDisplay={isFormDisplay}
                        agreementText={agreementText}
                    />
                </PlaceholderControl>

                <InspectorControls>
                    <FieldSettingsFill>
                        <PanelRow>
                            <TextControl
                                label={__('Checkbox Label', 'give')}
                                value={checkboxLabel}
                                onChange={(value) => setAttributes({checkboxLabel: value})}
                            />
                        </PanelRow>
                        <PanelRow>
                            <SelectControl
                                label={__('Display Type', 'give')}
                                onChange={(value) => setAttributes({displayType: value})}
                                value={displayType}
                                options={[
                                    {label: __('Show terms in modal', 'give'), value: 'showModalTerms'},
                                    {label: __('Show terms in form', 'give'), value: 'showFormTerms'},
                                    {label: __('Link to terms', 'give'), value: 'showLinkTerms'},
                                ]}
                                help={''}
                            />
                        </PanelRow>

                        {!isFormDisplay && (
                            <PanelRow>
                                <TextControl
                                    label={__('Link Text', 'give')}
                                    value={linkText}
                                    onChange={(value) => setAttributes({linkText: value})}
                                />
                            </PanelRow>
                        )}

                        {isLinkDisplay && (
                            <PanelRow>
                                <TextControl
                                    label={__('URL', 'give')}
                                    value={linkUrl}
                                    onChange={(value) => setAttributes({linkUrl: value})}
                                />
                            </PanelRow>
                        )}

                        {!isLinkDisplay && (
                            <PanelRow>
                                <ControlForPopover
                                    id="consent-block-popover-control"
                                    help={__(
                                        'This is the actual text which the user will have to agree to in order to make a donation.',
                                        'give'
                                    )}
                                    heading={__('Agreement Text', 'give')}
                                    onButtonClick={() => setShowAgreementTextModal(!showAgreementTextModal)}
                                    isButtonActive={showAgreementTextModal}
                                >
                                    <StyledPopover
                                        title={__('Agreement Text', 'give')}
                                        visible={showAgreementTextModal}
                                        onClose={() => setShowAgreementTextModal(false)}
                                    >
                                        <ClassicEditor
                                            id={'givewp-ffm-agreement-text'}
                                            label={__('', 'give')}
                                            content={agreementText}
                                            setContent={(value) => setAttributes({agreementText: value})}
                                        />
                                    </StyledPopover>
                                </ControlForPopover>
                            </PanelRow>
                        )}

                        {isModalDisplay && (
                            <>
                                <PanelRow>
                                    <TextControl
                                        label={__('Modal Heading', 'give')}
                                        value={modalHeading}
                                        onChange={(value) => setAttributes({modalHeading: value})}
                                    />
                                </PanelRow>
                                <PanelRow>
                                    <TextControl
                                        label={__('Modal Accept Button', 'give')}
                                        value={modalAcceptanceText}
                                        onChange={(value) => setAttributes({modalAcceptanceText: value})}
                                    />
                                </PanelRow>
                            </>
                        )}
                    </FieldSettingsFill>
                </InspectorControls>
            </>
        );
    },
};

type CheckboxProps = {
    label: string;
    linkText: string;
    isFormDisplay: boolean;
    agreementText: string;
};

function CheckboxPlaceholder({label, linkText, isFormDisplay, agreementText}: CheckboxProps) {
    return (
        <div className={'give-ffm-consent__checkbox'}>
            <div
                className={'give-ffm-consent__checkbox-wrapper'}
                style={{display: isFormDisplay ? 'block' : 'inline-flex'}}
            >
                <CheckboxControl label={label} onChange={null} disabled={true} />

                {isFormDisplay && (
                    <div className={'give-ffm-consent__checkbox__form-display'}>
                        <Markup content={agreementText} />
                    </div>
                )}

                {!isFormDisplay && <div className={'give-ffm-consent__checkbox__link'}>{linkText}</div>}
            </div>
        </div>
    );
}

const consent = {
    name: settings.name,
    settings,
};

export default consent;
