<?php

namespace GiveFormFieldManager\FormExtension\DonationForm\Actions;

use Give\DonationForms\Rules\ArrayRule;
use Give\Framework\Blocks\BlockModel;
use Give\Framework\FieldsAPI\Checkbox;
use Give\Framework\FieldsAPI\Consent;
use Give\Framework\FieldsAPI\Contracts\Node;
use Give\Framework\FieldsAPI\Date;
use Give\Framework\FieldsAPI\Email;
use Give\Framework\FieldsAPI\Exceptions\EmptyNameException;
use Give\Framework\FieldsAPI\Field;
use Give\Framework\FieldsAPI\File;
use Give\Framework\FieldsAPI\Hidden;
use Give\Framework\FieldsAPI\Html;
use Give\Framework\FieldsAPI\MultiSelect;
use Give\Framework\FieldsAPI\Phone;
use Give\Framework\FieldsAPI\Radio;
use Give\Framework\FieldsAPI\Select;
use Give\Framework\FieldsAPI\Textarea;
use Give\Framework\FieldsAPI\Url;
use Give\Vendors\StellarWP\Validation\Rules\In;
use GiveFormFieldManager\FormExtension\DonationForm\Rules\CheckboxFieldRule;
use GiveFormFieldManager\FormExtension\DonationForm\Rules\ConsentFieldRule;
use GiveFormFieldManager\FormExtension\DonationForm\Rules\MultiSelectFieldRule;
use GiveFormFieldManager\FormExtension\DonationForm\Rules\UrlRule;

class ConvertBlocksInForm
{
    /**
     * Converts the FFM blocks in the form to GiveWP Field API fields
     *
     * @since 3.1.1 update Radio and MultiSelect fields to use the In rule class
     * @since 3.1.0 added additional validation to Select, MultiSelect, Dropdown, Radio, Consent fields
     * @since 3.0.4 set default value fallback for fields with options
     * @since 3.0.3 return $node instead of null by default
     * @since 3.0.0
     *
     * @param Node|null $node
     * @param BlockModel $block
     * @param int $index
     *
     * @return Node|null
     * @throws EmptyNameException
     */
    public function __invoke(?Node $node, BlockModel $block, int $index): ?Node
    {
        switch ($block->name) {
            case 'givewp-form-field-manager/checkbox':
                return $this->applyFieldSettings(Checkbox::make($block->getAttribute('fieldName')), $block)
                    ->value($block->getAttribute('value'))
                    ->checked($block->getAttribute('checked'))
                    ->tap(function (Checkbox $field) use ($block) {
                        if ($block->getAttribute('value')) {
                            $field->rules(new CheckboxFieldRule($block->getAttribute('value')));
                        } else {
                            $field->rules('boolean');
                        }
                    });
            case 'givewp-form-field-manager/consent':
                return $this->applyFieldSettings(Consent::make($block->getAttribute('fieldName')), $block)
                    ->useGlobalSettings(false)
                    ->checkboxLabel($block->getAttribute('checkboxLabel'))
                    ->displayType($block->getAttribute('displayType'))
                    ->linkText($block->getAttribute('linkText'))
                    ->linkUrl($block->getAttribute('linkUrl'))
                    ->modalHeading($block->getAttribute('modalHeading'))
                    ->modalAcceptanceText($block->getAttribute('modalAcceptanceText'))
                    ->agreementText($block->getAttribute('agreementText'))
                    ->rules(new ConsentFieldRule());
            case 'givewp-form-field-manager/date':
                return $this->applyFieldSettings(Date::make($block->getAttribute('fieldName')), $block)
                    ->dateFormat($block->getAttribute('dateFormat'));
            case 'givewp-form-field-manager/dropdown':
                return $this->applyFieldSettings(Select::make($block->getAttribute('fieldName')), $block)
                    ->tap(function (Select $field) use ($block) {
                        ['options' => $options, 'checked' => $checked, 'optionsForRules' => $optionsForRules] = $this->prepareOptionsArray($block);

                        $field->options(...$options);

                        if (!empty($checked)) {
                            $field->defaultValue(current($checked));
                        }

                        $field->rules(new In(...$optionsForRules));
                    });
            case 'givewp-form-field-manager/email':
                return $this->applyFieldSettings(Email::make($block->getAttribute('fieldName')), $block)
                    ->defaultValue($block->getAttribute('defaultValue'))
                    ->rules('email');
            case 'givewp-form-field-manager/file-upload':
                return $this->applyFieldSettings(File::make($block->getAttribute('fieldName')), $block)
                    ->maxUploadSize(
                        $block->getAttribute('maxFileSize') > 0
                            ? $block->getAttribute(
                                'maxFileSize'
                            ) * 1024 // Converted KB to bytes
                            : wp_max_upload_size()
                    )
                    ->tap(function ($field) use ($block) {
                        $fileMimeTypes = [
                            'image' => ['image/jpeg', 'image/png', 'image/gif', 'image/bmp'],
                            'audio' => [
                                'audio/mpeg',
                                'audio/wav',
                                'audio/ogg',
                                'audio/x-ms-wma',
                                'audio/x-matroska',
                                'audio/mp4',
                                'audio/x-realaudio',
                                'audio/midi',
                                'audio/midi'
                            ],
                            'video' => [
                                'video/x-msvideo',
                                'video/divx',
                                'video/x-flv',
                                'video/quicktime',
                                'video/ogg',
                                'video/x-matroska',
                                'video/mp4',
                                'video/x-m4v',
                                'video/divx',
                                'video/mpeg',
                                'video/mpeg',
                                'video/mpeg'
                            ],
                            'pdf' => ['application/pdf'],
                            'csv' => ['text/csv'],
                            'exe' => ['application/x-msdownload'],
                            'zip' => [
                                'application/zip',
                                'application/gx-zip',
                                'application/x-rar-compressed',
                                'application/gzip',
                                'application/x-7z-compressed'
                            ],
                            'office' => [
                                'application/msword',
                                'application/vnd.ms-powerpoint',
                                'application/vnd.ms-powerpoint',
                                'application/vnd.ms-excel',
                                'application/msaccess',
                                'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
                                'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                                'application/vnd.openxmlformats-officedocument.presentationml.presentation',
                                'application/vnd.oasis.opendocument.text',
                                'application/vnd.oasis.opendocument.presentation',
                                'application/vnd.oasis.opendocument.graphics',
                                'application/vnd.oasis.opendocument.chart',
                                'application/vnd.oasis.opendocument.database',
                                'application/vnd.oasis.opendocument.formula',
                                'application/rtf',
                                'text/plain'
                            ],
                        ];

                        $allowedTypes = array_filter(
                            array_map(
                                function ($item) use ($fileMimeTypes) {
                                    return array_key_exists($item, $fileMimeTypes) ? $fileMimeTypes[$item] : [];
                                },
                                $block->getAttribute('allowedFileTypes')
                            )
                        );

                        if ($allowedTypes) {
                            $field->allowedMimeTypes(array_values(array_unique(array_merge(...$allowedTypes))));
                        }
                    });
            case 'givewp-form-field-manager/hidden':
                return $this->applyFieldSettings(Hidden::make($block->getAttribute('fieldName')), $block)
                    ->defaultValue($block->getAttribute('defaultValue') ?? '');
            case 'givewp-form-field-manager/html':
                return Html::make($block->getAttribute('fieldName'))
                    ->html($block->getAttribute('htmlCode'));
            case 'givewp-form-field-manager/multi-select':
                return $this->applyFieldSettings(MultiSelect::make($block->getAttribute('fieldName')), $block)
                    ->fieldType($block->getAttribute('fieldType'))
                    ->tap(function (MultiSelect $field) use ($block) {
                        ['options' => $options, 'checked' => $checked, 'optionsForRules' => $optionsForRules] = $this->prepareOptionsArray($block);

                        $field->options(...$options);
                        $field->defaultValue($checked ?? []);
                        $field->rules(new MultiSelectFieldRule($optionsForRules), new ArrayRule(true));
                    });
            case 'givewp-form-field-manager/phone':
                return $this->applyFieldSettings(Phone::make($block->getAttribute('fieldName')), $block)
                    ->phoneFormat($block->getAttribute('phoneFormat'))
                    ->defaultValue($block->getAttribute('defaultValue'));
            case 'givewp-form-field-manager/radio':
                return $this->applyFieldSettings(Radio::make($block->getAttribute('fieldName')), $block)
                    ->tap(function (Radio $field) use ($block) {
                        ['options' => $options, 'checked' => $checked, 'optionsForRules' => $optionsForRules] = $this->prepareOptionsArray($block);

                        $field->options(...$options);

                        $field->defaultValue(!empty(current($checked)) ? current($checked) : $options[0][0]);

                        $field->rules(new In(...$optionsForRules));
                    });
            case 'givewp-form-field-manager/textarea':
                return $this->applyFieldSettings(Textarea::make($block->getAttribute('fieldName')), $block)
                    ->defaultValue($block->getAttribute('defaultValue'))
                    ->rows($block->getAttribute('rows'));
            case 'givewp-form-field-manager/url':
                return $this->applyFieldSettings(Url::make($block->getAttribute('fieldName')), $block)
                    ->defaultValue($block->getAttribute('defaultValue'))
                    ->rules(new UrlRule());
        }

        return $node;
    }

    /**
     * Applies the standard field settings to each field. Just keeping things dry.
     *
     * @template T of Field
     *
     * @since 3.0.0
     * @param T $field
     *
     * @return T
     *
     */
    private function applyFieldSettings(Field $field, BlockModel $block): Field
    {
        $field->required($block->getAttribute('isRequired') === true);
        $field->storeAsDonorMeta($block->getAttribute('storeAsDonorMeta') === true);
        $field->showInAdmin($block->getAttribute('displayInAdmin') === true);
        $field->showInReceipt($block->getAttribute('displayInReceipt') === true);

        if (method_exists($field, 'label')) {
            $field->label($block->getAttribute('label') ?? '');
        }

        if (method_exists($field, 'placeholder')) {
            $field->placeholder($block->getAttribute('placeholder') ?? '');
        }

        if (method_exists($field, 'description')) {
            $field->description($block->getAttribute('description') ?? '');
        }

        return $field;
    }


    /**
     * Prepares the options array to be used in the field.
     *
     * @since 3.0.3 refactored from setFieldOptions to only return the options array
     * @since 3.0.0
     *
     * @return array ['options' => [], 'checked' => []]
     */
    private function prepareOptionsArray(BlockModel $block): array
    {
        $checked = [];
        $options = array_values(
            array_filter(
                array_map(
                    function ($item) use (&$checked) {
                        $option = [$item['value'] ?: $item['label'], $item['label']];

                        if ($item['checked']) {
                            $checked[] = $option[0];
                        }

                        return $option;
                    },
                    $block->getAttribute('options')
                ),
                function ($item) {
                    return $item[0] !== '';
                }
            )
        );

        return [
            'options' => $options,
            'checked' => $checked,
            'optionValues' => array_column($options, 0),
            'optionLabels' => array_column($options, 1),
            'optionsForRules' => array_unique(array_merge(...$options)),
        ];
    }
}
