<?php 
/**
 * EVOST WC Block
 * @version 1.2.4
 */

class EVOST_WC_Block{
 	
	public function __construct() {
        
        // Store API Hooks
        add_action('woocommerce_store_api_cart_update_from_request', [$this, 'store_api_cart_update'], 10, 2);
        add_action('woocommerce_store_api_checkout_order_processed', [$this, 'store_api_order_processed'], 10, 1);
        add_action('wp_enqueue_scripts', [$this, 'enqueue_block_checkout_scripts'], 10);
        add_action('woocommerce_store_api_register_endpoint_data', [$this, 'register_store_api_seat_data'], 10);
    }

    // Register seat data for Store API cart endpoint
    public function register_store_api_seat_data() {
        woocommerce_store_api_register_endpoint_data([
            'endpoint' => 'cart',
            'namespace' => 'evost',
            'data_callback' => function($cart) {
                $data = [];
                foreach ($cart->get_cart() as $cart_item_key => $cart_item) {
                    if (isset($cart_item['evost_data'])) {
                        $data[$cart_item_key] = [
                            'seat_slug' => $cart_item['evost_data']['seat_slug'] ?? '',
                            'seat_number' => $cart_item['evost_data']['seat_number'] ?? '',
                            'seat_type' => $cart_item['evost_data']['seat_type'] ?? '',
                            'seat_price' => $cart_item['evost_data']['seat_price'] ?? '',
                            'seat_time' => $cart_item['evost_data']['seat_time'] ?? '',
                        ];
                    }
                }
                return $data;
            },
            'schema_callback' => function() {
                return [
                    'seat_slug' => ['description' => 'Seat slug', 'type' => 'string'],
                    'seat_number' => ['description' => 'Seat number', 'type' => 'string'],
                    'seat_type' => ['description' => 'Seat type', 'type' => 'string'],
                    'seat_price' => ['description' => 'Seat price', 'type' => 'string'],
                    'seat_time' => ['description' => 'Seat expiration time', 'type' => 'integer'],
                ];
            },
        ]);
    }

    // Enqueue scripts for Block Checkout
    public function enqueue_block_checkout_scripts() {
        if (!is_checkout()) return;

        wp_enqueue_script(
            'evost-block-checkout',
            '',
            ['wp-element', 'wc-blocks-checkout'],
            '1.0.0',
            true
        );

        // Localize seat data for frontend
        $seat_data = [];
        foreach (WC()->cart->get_cart() as $cart_item_key => $cart_item) {
            if (isset($cart_item['evost_data']['seat_time'])) {
                $SEAT = new EVOST_Seats_Seat(
                    $cart_item['evotx_event_id_wc'],
                    $cart_item['product_id'],
                    $cart_item['evost_data']['seat_slug']
                );
                $seat_data[$cart_item_key] = [
                    'expiration_time' => $SEAT->get_seat_expiration_time(
                        $cart_item['evost_data']['seat_slug'],
                        $cart_item_key
                    ),
                    'seat_slug' => $cart_item['evost_data']['seat_slug'],
                ];
            }
        }
        // Only add inline script if there’s seat data
        if (!empty($seat_data)) {
            wp_localize_script('evost-block-checkout', 'evostSeatData', $seat_data);

            // Inline JavaScript
            ob_start();?>
            
                document.addEventListener('DOMContentLoaded', () => {
                    if (typeof evostSeatData !== 'undefined') {
                        Object.keys(evostSeatData).forEach(cartItemKey => {
                            const seat = evostSeatData[cartItemKey];
                            console.log(`Seat ${seat.seat_slug} (Number: ${seat.seat_number}, Type: ${seat.seat_type}) expires at ${new Date(seat.expiration_time * 1000)}`);

                            // Example: Display seat details in Block Checkout UI
                            const seatInfo = document.createElement('div');
                            seatInfo.className = 'evost-seat-info';
                            seatInfo.style.marginTop = '10px';
                            seatInfo.style.fontSize = '14px';
                            seatInfo.textContent = `Seat: ${seat.seat_number || seat.seat_slug} (Expires in ${Math.max(0, seat.expiration_time - Math.floor(Date.now() / 1000))} seconds)`;
                            const checkoutContainer = document.querySelector('.wc-block-checkout__main') || document.body;
                            checkoutContainer.appendChild(seatInfo);

                            // TODO: Customize further, e.g., add countdown timer or styling
                            // Example countdown timer:
                            /*
                            setInterval(() => {
                                const secondsLeft = Math.max(0, seat.expiration_time - Math.floor(Date.now() / 1000));
                                seatInfo.textContent = `Seat: ${seat.seat_number || seat.seat_slug} (Expires in ${secondsLeft} seconds)`;
                                if (secondsLeft <= 0) {
                                    seatInfo.textContent = `Seat: ${seat.seat_number || seat.seat_slug} (Expired)`;
                                }
                            }, 1000);
                            */
                        });
                    }
                });
            
            <?php 

            $inline_js = ob_get_clean();
            wp_add_inline_script('evost-block-checkout', $inline_js);
        }
    }

    // Validate and update cart items in Store API
    public function store_api_cart_update($cart, $request) {
        foreach ($cart->get_cart() as $cart_item_key => $cart_item) {
            if (!isset($cart_item['evost_data']['seat_slug'])) continue;

            $seat_slug = $cart_item['evost_data']['seat_slug'];
            $SEAT = new EVOST_Seats_Seat(
                $cart_item['evotx_event_id_wc'],
                $cart_item['product_id'],
                $seat_slug
            );

            // Validate seat availability
            if (!$SEAT->is_seat_available($cart_item['quantity'])) {
                $cart->remove_cart_item($cart_item_key);
                wc_add_notice('Ticket removed from cart, seat no longer available for sale!', 'error');
                continue;
            }

            // Check seat expiration
            if ($SEAT->has_seat_expired($cart_item_key, $seat_slug)) {
                $cart->remove_cart_item($cart_item_key);
                wc_add_notice('Ticket removed from cart due to seat expiration time!', 'error');
                $SEAT->restock_temphold_seat($cart_item_key, $cart_item['quantity'], $seat_slug);
            } else {
                $SEAT->reset_expiration_time_from_cart($cart_item_key, $seat_slug, $cart_item['quantity']);
            }
        }
    }

    // Process seat stock during Store API order creation
    public function store_api_order_processed($order) {
        if (!($order instanceof WC_Order)) return;

        foreach ($order->get_items() as $item_id => $item) {
            $seat_slug = wc_get_order_item_meta($item_id, '_evost_seat_slug');
            if (!$seat_slug) continue;

            $event_id = $item->get_product()->get_meta('_eventid');
            if (!$event_id) continue;

            $TIX_EVENT = new EVO_Event($event_id, '', 0);
            $SEAT = new EVOST_Seats_Seat($TIX_EVENT, $item['product_id'], $seat_slug);

            $SEAT->delete_seat_expiration($seat_slug);

            if ($SEAT->seat_type == 'seat') {
                $SEAT->adjust_stock('reduce'); // Mark seat as unavailable
                $order->add_order_note(sprintf(
                    __('Event: (%s) seat %s made unavailable.', 'woocommerce'),
                    $TIX_EVENT->get_title(),
                    $SEAT->get_seat_number()
                ));
            }
        }
    }
 } 