<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

if (!class_exists('Give')) {
    return;
}

use Elementor\Controls_Manager;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;

class Humani_Elementor_Give_Campain extends \Elementor\Widget_Base {

    public function get_name() {
        return 'humani-give-campain';
    }

    public function get_title() {
        return __('Humani Give Form Grid', 'humani');
    }

    public function get_categories() {
        return array('humani-addons');
    }

    public function get_script_depends() {
        return ['humani-elementor-give-form-grid', 'slick'];
    }

    protected function register_controls() {
        $this->start_controls_section(
            'section_give_form',
            [
                'label' => __('Give Form Grid', 'humani'),
            ]
        );

        $this->add_control(
            'form_id',
            [
                'label'       => __('Form ids', 'humani'),
                'type'        => Controls_Manager::SELECT2,
                'options'     => $this->get_form_ids(),
                'multiple'    => true,
                'description' => __('Enter a comma-separated list of form IDs. If empty, all published forms are displayed.', 'humani'),
            ]
        );

        $this->add_control(
            'excluded_form_id',
            [
                'label'       => __('Excluded Form IDs', 'humani'),
                'type'        => Controls_Manager::SELECT2,
                'options'     => $this->get_form_ids(),
                'multiple'    => true,
                'description' => __('Enter a comma-separated list of form IDs to exclude those from the grid.', 'humani'),
            ]
        );
        $enable_category = give_is_setting_enabled(give_get_option('categories', 'disabled'));
        if ($enable_category) {
            $this->add_control(
                'cats',
                [
                    'label'    => __('Category IDs', 'humani'),
                    'type'     => Controls_Manager::SELECT2,
                    'options'  => $this->get_form_taxonomy(),
                    'multiple' => true,
                ]
            );
        }

        $this->add_responsive_control(
            'columns',
            [
                'label'       => __('Columns', 'humani'),
                'type'        => Controls_Manager::SELECT,
                'options'     => [
                    '1' => __('1', 'humani'),
                    '2' => __('2', 'humani'),
                    '3' => __('3', 'humani'),
                    '4' => __('4', 'humani'),
                    '5' => __('5', 'humani'),
                    '6' => __('6', 'humani'),
                ],
                'default'     => 4,
                'description' => __('Sets the number of forms per row.', 'humani'),
            ]
        );
        $this->add_control(
            'show_goal',
            [
                'label'       => __('Show Goal', 'humani'),
                'type'        => Controls_Manager::SWITCHER,
                'label_on'    => 'Hide',
                'label_off'   => 'Show',
                'default'     => 'yes',
                'description' => __('Do you want to display the goal\'s progress bar?', 'humani'),
            ]
        );
        $this->add_control(
            'show_excerpt',
            [
                'label'     => __('Show Excerpt', 'humani'),
                'type'      => Controls_Manager::SWITCHER,
                'label_on'  => 'Hide',
                'label_off' => 'Show',
                'default'   => 'yes',
            ]
        );
        $this->add_control(
            'show_featured_image',
            [
                'label'     => __('Show Featured Image', 'humani'),
                'type'      => Controls_Manager::SWITCHER,
                'label_on'  => 'Hide',
                'label_off' => 'Show',
                'default'   => 'yes',
            ]
        );

        $this->add_control(
            'display_style',
            [
                'label'   => __('display_style', 'humani'),
                'type'    => Controls_Manager::SELECT,
                'options' => [
                    'redirect'     => __('Redirect', 'humani'),
                    'modal_reveal' => __('Modal', 'humani'),
                ],
                'default' => 'redirect'
            ]
        );

        $this->add_control(
            'forms_per_page',
            [
                'label'   => __('Forms Per Page', 'humani'),
                'type'    => Controls_Manager::NUMBER,
                'default' => 12,
            ]
        );

        $this->add_control(
            'show_paged',
            [
                'label'     => __('Enables/Disables pagination', 'humani'),
                'type'      => Controls_Manager::SWITCHER,
                'label_on'  => 'Hide',
                'label_off' => 'Show',
                'default'   => 'yes',
            ]
        );

        $this->add_control(
            'excerpt_length',
            [
                'label'   => __('Excerpt Length', 'humani'),
                'type'    => Controls_Manager::NUMBER,
                'default' => 25,
            ]
        );

        $this->add_responsive_control(
            'item_spacing',
            [
                'label'      => esc_html__('Spacing', 'humani'),
                'type'       => Controls_Manager::SLIDER,
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'size_units' => ['px', 'em'],
                'selectors'  => [
                    '{{WRAPPER}} .give-wrap .give-grid'         => 'grid-gap: {{SIZE}}{{UNIT}}',
                ],
            ]
        );

        $this->add_control('block_style', [
            'label'   => esc_html__('Block Style', 'humani'),
            'type'    => Controls_Manager::SELECT,
            'options' => [
                'block-1' => __('Style 1', 'humani'),
                'block-2' => __('Style 2', 'humani'),
                'block-3' => __('Style 3', 'humani'),
                'block-4' => __('Style 4', 'humani'),
            ],
            'default' => 'block-1'
        ]);

        $this->end_controls_section();

        $this->start_controls_section(
            'section_style_wrap',
            [
                'label' => esc_html__('Wrap', 'humani'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'label' => esc_html__('Box Shadow', 'humani'),
                'name' => 'wrap_box_shadow',
                'selector' => '{{WRAPPER}} .give-wrap .give-card',
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'label' => esc_html__('Box Shadow Hover', 'humani'),
                'name' => 'wrap_box_shadow_hover',
                'selector' => '{{WRAPPER}} .give-wrap .give-card:hover',
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'wrap_border',
                'selector' => '{{WRAPPER}} .give-wrap .give-card',
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'wrap_border_radius',
            [
                'label' => __( 'Border Radius', 'humani' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .give-wrap .give-card' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'wrapper_padding_inner',
            [
                'label'      => esc_html__('Padding', 'humani'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .give-wrap .give-card__body' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        $this->add_control_carousel();

    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $atts     = [
            'columns'             => $settings['columns'],
            'show_goal'           => $settings['show_goal'],
            'show_excerpt'        => $settings['show_excerpt'],
            'show_featured_image' => $settings['show_featured_image'],
            'display_style'       => $settings['display_style'],
            'forms_per_page'      => $settings['forms_per_page'],
            'paged'               => $settings['show_paged'],
            'excerpt_length'      => $settings['excerpt_length'],
            'image_size'          => 'givelove-featured-image-large'
        ];
        if ($settings['columns'] <= 2) {
            $atts['image_size'] = 'full';
        }
        if (!empty($settings['form_id'])) {
            $atts['ids'] = implode(',', $settings['form_id']);
        }
        if (!empty($settings['form_id'])) {
            $atts['exclude'] = implode(',', $settings['excluded_form_id']);
        }
        if (!empty($settings['cats'])) {
            $atts['cats'] = implode(',', $settings['cats']);
        }
        $code = '';
        foreach ($atts as $key => $value) {
            $code .= $key . '="' . (empty($value) ? 'false' : $value) . '" ';
        }

        if ($settings['enable_carousel'] === 'yes') {
            $this->add_render_attribute('wrap', 'class', 'humani-carousel-active');
            $carousel_settings = $this->get_carousel_settings();
            $this->add_render_attribute('wrap', 'data-settings', wp_json_encode($carousel_settings));
            echo '<div ' . $this->get_render_attribute_string('wrap') . '>';
        }
        add_action('humani_blog_style', [$this, 'filter_blog_style'], 99);
        echo do_shortcode('[give_form_grid ' . $code . ' ]');
        remove_action('humani_blog_style', [$this, 'filter_blog_style'], 99);
        if ($settings['enable_carousel'] === 'yes') {
            echo '</div>';
        }
    }

    public function filter_blog_style() {
        $settings = $this->get_settings_for_display();
        return $settings['block_style'] ? $settings['block_style'] : 'block-1';
    }

    private function get_form_ids() {
        $args          = array(
            'post_type'      => 'give_forms',
            'posts_per_page' => -1,
            'post_status'    => 'publish',
        );
        $give_form_ids = array();
        $give_forms    = get_posts($args);
        foreach ($give_forms as $give_form) {
            $form_title                    = empty($give_form->post_title) ? sprintf(__('Untitled (#%s)', 'humani'), $give_form->ID) : $give_form->post_title;
            $give_form_ids[$give_form->ID] = $form_title;
        }
        return $give_form_ids;
    }

    private function get_form_taxonomy() {
        $enable_category = give_is_setting_enabled(give_get_option('categories', 'disabled'));
        if ($enable_category) {
            $args               = array(
                'hide_empty' => false,
                'orderby'    => 'name',
                'order'      => 'ASC',
                'number'     => 0,
            );
            $terms              = get_terms('give_forms_category', $args);
            $give_form_taxonomy = array();
            foreach ($terms as $term) {
                $give_form_taxonomy[$term->term_id] = $term->name;
            }
            return $give_form_taxonomy;
        }
    }

    protected function add_control_carousel($condition = array()) {
        $this->start_controls_section(
            'section_carousel_options',
            [
                'label'     => esc_html__('Carousel Options', 'humani'),
                'type'      => Controls_Manager::SECTION,
                'condition' => $condition,
            ]
        );

        $this->add_control(
            'enable_carousel',
            [
                'label' => esc_html__('Enable', 'humani'),
                'type'  => Controls_Manager::SWITCHER,
            ]
        );

        $this->add_control(
            'carousel_visible',
            [
                'label'        => esc_html__('Visible', 'humani'),
                'type'         => Controls_Manager::SELECT,
                'default'      => '',
                'options'      => [
                    ''        => esc_html__('Default', 'humani'),
                    'visible' => esc_html__('Visible', 'humani'),
                    'left'    => esc_html__('left', 'humani'),
                    'right'   => esc_html__('right', 'humani'),
                ],
                'condition'    => [
                    'enable_carousel' => 'yes'
                ],
                'prefix_class' => 'carousel-visible-',
            ]
        );


        $this->add_control(
            'navigation',
            [
                'label'     => esc_html__('Navigation', 'humani'),
                'type'      => Controls_Manager::SELECT,
                'default'   => 'dots',
                'options'   => [
                    'both'   => esc_html__('Arrows and Dots', 'humani'),
                    'arrows' => esc_html__('Arrows', 'humani'),
                    'dots'   => esc_html__('Dots', 'humani'),
                    'none'   => esc_html__('None', 'humani'),
                ],
                'condition' => [
                    'enable_carousel' => 'yes'
                ],
            ]
        );

        $this->add_control(
            'pause_on_hover',
            [
                'label'     => esc_html__('Pause on Hover', 'humani'),
                'type'      => Controls_Manager::SWITCHER,
                'default'   => 'yes',
                'condition' => [
                    'enable_carousel' => 'yes'
                ],
            ]
        );

        $this->add_control(
            'autoplay',
            [
                'label'     => esc_html__('Autoplay', 'humani'),
                'type'      => Controls_Manager::SWITCHER,
                'default'   => 'yes',
                'condition' => [
                    'enable_carousel' => 'yes'
                ],
            ]
        );

        $this->add_control(
            'autoplay_speed',
            [
                'label'     => esc_html__('Autoplay Speed', 'humani'),
                'type'      => Controls_Manager::NUMBER,
                'default'   => 5000,
                'condition' => [
                    'autoplay'        => 'yes',
                    'enable_carousel' => 'yes'
                ],
                'selectors' => [
                    '{{WRAPPER}} .slick-slide-bg' => 'animation-duration: calc({{VALUE}}ms*1.2); transition-duration: calc({{VALUE}}ms)',
                ],
            ]
        );

        $this->add_control(
            'infinite',
            [
                'label'     => esc_html__('Infinite Loop', 'humani'),
                'type'      => Controls_Manager::SWITCHER,
                'default'   => 'yes',
                'condition' => [
                    'enable_carousel' => 'yes'
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'carousel_arrows',
            [
                'label'      => esc_html__('Carousel Arrows', 'humani'),
                'conditions' => [
                    'relation' => 'and',
                    'terms'    => [
                        [
                            'name'     => 'enable_carousel',
                            'operator' => '==',
                            'value'    => 'yes',
                        ],
                        [
                            'name'     => 'navigation',
                            'operator' => '!==',
                            'value'    => 'none',
                        ],
                        [
                            'name'     => 'navigation',
                            'operator' => '!==',
                            'value'    => 'dots',
                        ],
                    ],
                ],
            ]
        );

        //Style arrow
        $this->add_control(
            'style_arrow',
            [
                'label'        => esc_html__('Style Arrow', 'humani'),
                'type'         => Controls_Manager::SELECT,
                'options'      => [
                    'style-1' => esc_html__('Style 1', 'humani'),
                    'style-2' => esc_html__('Style 2', 'humani'),
                    'style-3' => esc_html__('Style 3', 'humani'),
                ],
                'default'      => 'style-1',
                'prefix_class' => 'arrow-'
            ]
        );

        //add icon next size
        $this->add_responsive_control(
            'icon_size',
            [
                'label'     => esc_html__('Size', 'humani'),
                'type'      => Controls_Manager::SLIDER,
                'range'     => [
                    'px' => [
                        'min' => 0,
                        'max' => 300,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .slick-arrow:before' => 'font-size: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        //add icon next color
        $this->add_control(
            'color_button',
            [
                'label' => esc_html__('Color', 'humani'),
                'type'  => Controls_Manager::HEADING,
            ]
        );

        $this->start_controls_tabs('tabs_carousel_arrow_style');

        $this->start_controls_tab(
            'tab_carousel_arrow_normal',
            [
                'label' => esc_html__('Normal', 'humani'),
            ]
        );

        $this->add_control(
            'carousel_arrow_color_icon',
            [
                'label'     => esc_html__('Color icon', 'humani'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .slick-slider button.slick-prev:before' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .slick-slider button.slick-next:before' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'carousel_arrow_color_border',
            [
                'label'     => esc_html__('Color Border', 'humani'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .slick-slider button.slick-prev' => 'border-color: {{VALUE}};',
                    '{{WRAPPER}} .slick-slider button.slick-next' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'carousel_arrow_color_background',
            [
                'label'     => esc_html__('Color background', 'humani'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .slick-slider button.slick-prev' => 'background-color: {{VALUE}};',
                    '{{WRAPPER}} .slick-slider button.slick-next' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'tab_carousel_arrow_hover',
            [
                'label' => esc_html__('Hover', 'humani'),
            ]
        );

        $this->add_control(
            'carousel_arrow_color_icon_hover',
            [
                'label'     => esc_html__('Color icon', 'humani'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .slick-slider button.slick-prev:hover:before' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .slick-slider button.slick-next:hover:before' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'carousel_arrow_color_border_hover',
            [
                'label'     => esc_html__('Color Border', 'humani'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .slick-slider button.slick-prev:hover' => 'border-color: {{VALUE}};',
                    '{{WRAPPER}} .slick-slider button.slick-next:hover' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'carousel_arrow_color_background_hover',
            [
                'label'     => esc_html__('Color background', 'humani'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .slick-slider button.slick-prev:hover' => 'background-color: {{VALUE}};',
                    '{{WRAPPER}} .slick-slider button.slick-next:hover' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_control(
            'next_heading',
            [
                'label' => esc_html__('Next button', 'humani'),
                'type'  => Controls_Manager::HEADING,
            ]
        );

        $this->add_control(
            'next_vertical',
            [
                'label'       => esc_html__('Next Vertical', 'humani'),
                'type'        => Controls_Manager::CHOOSE,
                'label_block' => false,
                'options'     => [
                    'top'    => [
                        'title' => esc_html__('Top', 'humani'),
                        'icon'  => 'eicon-v-align-top',
                    ],
                    'bottom' => [
                        'title' => esc_html__('Bottom', 'humani'),
                        'icon'  => 'eicon-v-align-bottom',
                    ],
                ]
            ]
        );

        $this->add_responsive_control(
            'next_vertical_value',
            [
                'type'       => Controls_Manager::SLIDER,
                'show_label' => false,
                'size_units' => ['px', '%'],
                'range'      => [
                    'px' => [
                        'min'  => -1000,
                        'max'  => 1000,
                        'step' => 1,
                    ],
                    '%'  => [
                        'min' => -100,
                        'max' => 100,
                    ],
                ],
                'default'    => [
                    'unit' => '%',
                    'size' => 50,
                ],
                'selectors'  => [
                    '{{WRAPPER}} .slick-next' => 'top: unset; bottom: unset; {{next_vertical.value}}: {{SIZE}}{{UNIT}};',
                ]
            ]
        );
        $this->add_control(
            'next_horizontal',
            [
                'label'       => esc_html__('Next Horizontal', 'humani'),
                'type'        => Controls_Manager::CHOOSE,
                'label_block' => false,
                'options'     => [
                    'left'  => [
                        'title' => esc_html__('Left', 'humani'),
                        'icon'  => 'eicon-h-align-left',
                    ],
                    'right' => [
                        'title' => esc_html__('Right', 'humani'),
                        'icon'  => 'eicon-h-align-right',
                    ],
                ],
                'defautl'     => 'right'
            ]
        );
        $this->add_responsive_control(
            'next_horizontal_value',
            [
                'type'       => Controls_Manager::SLIDER,
                'show_label' => false,
                'size_units' => ['px', 'em', '%'],
                'range'      => [
                    'px' => [
                        'min'  => -1000,
                        'max'  => 1000,
                        'step' => 1,
                    ],
                    '%'  => [
                        'min' => -100,
                        'max' => 100,
                    ],
                ],
                'default'    => [
                    'unit' => 'px',
                    'size' => -45,
                ],
                'selectors'  => [
                    '{{WRAPPER}} .slick-next' => 'left: unset; right: unset;{{next_horizontal.value}}: {{SIZE}}{{UNIT}};',
                ]
            ]
        );

        $this->add_control(
            'prev_heading',
            [
                'label'     => esc_html__('Prev button', 'humani'),
                'type'      => Controls_Manager::HEADING,
                'separator' => 'before'
            ]
        );

        $this->add_control(
            'prev_vertical',
            [
                'label'       => esc_html__('Prev Vertical', 'humani'),
                'type'        => Controls_Manager::CHOOSE,
                'label_block' => false,
                'options'     => [
                    'top'    => [
                        'title' => esc_html__('Top', 'humani'),
                        'icon'  => 'eicon-v-align-top',
                    ],
                    'bottom' => [
                        'title' => esc_html__('Bottom', 'humani'),
                        'icon'  => 'eicon-v-align-bottom',
                    ],
                ]
            ]
        );

        $this->add_responsive_control(
            'prev_vertical_value',
            [
                'type'       => Controls_Manager::SLIDER,
                'show_label' => false,
                'size_units' => ['px', '%'],
                'range'      => [
                    'px' => [
                        'min'  => -1000,
                        'max'  => 1000,
                        'step' => 1,
                    ],
                    '%'  => [
                        'min' => -100,
                        'max' => 100,
                    ],
                ],
                'default'    => [
                    'unit' => '%',
                    'size' => 50,
                ],
                'selectors'  => [
                    '{{WRAPPER}} .slick-prev' => 'top: unset; bottom: unset; {{prev_vertical.value}}: {{SIZE}}{{UNIT}};',
                ]
            ]
        );
        $this->add_control(
            'prev_horizontal',
            [
                'label'       => esc_html__('Prev Horizontal', 'humani'),
                'type'        => Controls_Manager::CHOOSE,
                'label_block' => false,
                'options'     => [
                    'left'  => [
                        'title' => esc_html__('Left', 'humani'),
                        'icon'  => 'eicon-h-align-left',
                    ],
                    'right' => [
                        'title' => esc_html__('Right', 'humani'),
                        'icon'  => 'eicon-h-align-right',
                    ],
                ],
                'defautl'     => 'left'
            ]
        );
        $this->add_responsive_control(
            'prev_horizontal_value',
            [
                'type'       => Controls_Manager::SLIDER,
                'show_label' => false,
                'size_units' => ['px', 'em', '%'],
                'range'      => [
                    'px' => [
                        'min'  => -1000,
                        'max'  => 1000,
                        'step' => 1,
                    ],
                    '%'  => [
                        'min' => -100,
                        'max' => 100,
                    ],
                ],
                'default'    => [
                    'unit' => 'px',
                    'size' => -45,
                ],
                'selectors'  => [
                    '{{WRAPPER}} .slick-prev' => 'left: unset; right: unset; {{prev_horizontal.value}}: {{SIZE}}{{UNIT}};',
                ]
            ]
        );

        $this->end_controls_section();
    }

    protected function get_carousel_settings() {
        $settings = $this->get_settings_for_display();
        $breakpoints = \Elementor\Plugin::$instance->breakpoints->get_breakpoints();
        $tablet      = isset($settings['columns_tablet']) ? $settings['columns_tablet'] : 2;
        return array(
            'navigation'         => $settings['navigation'],
            'autoplayHoverPause' => $settings['pause_on_hover'] === 'yes' ? true : false,
            'autoplay'           => $settings['autoplay'] === 'yes' ? true : false,
            'autoplaySpeed'      => !empty($settings['autoplay_speed']) ? $settings['autoplay_speed'] : 5000,
            'items'              => $settings['columns'],
            'items_tablet'       => isset($settings['columns_tablet']) ? $settings['columns_tablet'] : $settings['columns'],
            'items_tablet_extra'      => isset($settings['columns_tablet_extra']) ? $settings['columns_tablet_extra'] : $settings['columns'],
            'items_laptop'            => isset($settings['columns_laptop']) ? $settings['columns_laptop'] : $settings['columns'],
            'items_mobile_extra'      => isset($settings['columns_mobile_extra']) ? $settings['columns_mobile_extra'] : $tablet,
            'items_mobile'       => isset($settings['columns_mobile']) ? $settings['columns_mobile'] : 1,
            'loop'               => $settings['infinite'] === 'yes' ? true : false,
            'breakpoint_laptop'       => $breakpoints['laptop']->get_value(),
            'breakpoint_tablet_extra' => $breakpoints['tablet_extra']->get_value(),
            'breakpoint_tablet'       => $breakpoints['tablet']->get_value(),
            'breakpoint_mobile_extra' => $breakpoints['mobile_extra']->get_value(),
            'breakpoint_mobile'       => $breakpoints['mobile']->get_value(),
        );
    }
}

$widgets_manager->register(new Humani_Elementor_Give_Campain());