<?php
/**
 * CakePHP(tm) : Rapid Development Framework (http://cakephp.org)
 * Copyright (c) Cake Software Foundation, Inc. (http://cakefoundation.org)
 * Licensed under The MIT License
 * Redistributions of files must retain the above copyright notice.
 *
 * @copyright     Copyright (c) Cake Software Foundation, Inc. (http://cakefoundation.org)
 * @link          http://cakephp.org CakePHP(tm) Project
 * @since         3.5.0
 * @license       http://www.opensource.org/licenses/mit-license.php MIT License
 */
namespace Cake\Test\TestCase\Http\Cookie;

use Cake\Chronos\Chronos;
use Cake\Http\Cookie\Cookie;
use Cake\TestSuite\TestCase;

/**
 * HTTP cookies test.
 */
class CookieTest extends TestCase
{
    /**
     * Generate invalid cookie names.
     *
     * @return array
     */
    public function invalidNameProvider()
    {
        return [
            ['no='],
            ["no\rnewline"],
            ["no\nnewline"],
            ["no\ttab"],
            ["no,comma"],
            ["no;semi"],
        ];
    }

    /**
     * Test invalid cookie name
     *
     * @dataProvider invalidNameProvider
     */
    public function testValidateNameInvalidChars($name)
    {
        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage('contains invalid characters.');
        new Cookie($name, 'value');
    }

    /**
     * Test empty cookie name
     *
     * @return void
     */
    public function testValidateNameEmptyName()
    {
        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage('The cookie name cannot be empty.');
        new Cookie('', '');
    }

    /**
     * Tests the header value
     *
     * @return void
     */
    public function testToHeaderValue()
    {
        $cookie = new Cookie('cakephp', 'cakephp-rocks');
        $result = $cookie->toHeaderValue();
        $this->assertEquals('cakephp=cakephp-rocks; path=/', $result);

        $date = Chronos::createFromFormat('m/d/Y h:i:s', '12/1/2027 12:00:00');

        $cookie = new Cookie('cakephp', 'cakephp-rocks');
        $cookie = $cookie->withDomain('cakephp.org')
            ->withExpiry($date)
            ->withHttpOnly(true)
            ->withSecure(true);
        $result = $cookie->toHeaderValue();

        $expected = 'cakephp=cakephp-rocks; expires=Wed, 01-Dec-2027 12:00:00 GMT; path=/; domain=cakephp.org; secure; httponly';
        $this->assertEquals($expected, $result);
    }

    /**
     * Test getting the value from the cookie
     *
     * @return void
     */
    public function testGetValue()
    {
        $cookie = new Cookie('cakephp', 'cakephp-rocks');
        $result = $cookie->getValue();
        $this->assertEquals('cakephp-rocks', $result);

        $cookie = new Cookie('cakephp', '');
        $result = $cookie->getValue();
        $this->assertEquals('', $result);
    }

    /**
     * Test setting values in cookies
     *
     * @return void
     */
    public function testWithValue()
    {
        $cookie = new Cookie('cakephp', 'cakephp-rocks');
        $new = $cookie->withValue('new');
        $this->assertNotSame($new, $cookie, 'Should make a clone');
        $this->assertSame('cakephp-rocks', $cookie->getValue(), 'old instance not modified');
        $this->assertSame('new', $new->getValue());
    }

    /**
     * Test getting the value from the cookie
     *
     * @return void
     */
    public function testGetStringValue()
    {
        $cookie = new Cookie('cakephp', 'thing');
        $this->assertSame('thing', $cookie->getStringValue());

        $value = ['user_id' => 1, 'token' => 'abc123'];
        $cookie = new Cookie('cakephp', $value);

        $this->assertSame($value, $cookie->getValue());
        $this->assertSame(json_encode($value), $cookie->getStringValue());
    }

    /**
     * Test setting domain in cookies
     *
     * @return void
     */
    public function testWithDomainInvalidConstructor()
    {
        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage('The provided arg must be of type `string` but `integer` given');
        new Cookie('cakephp', 'rocks', null, '', 1234);
    }

    /**
     * Test setting domain in cookies
     *
     * @return void
     */
    public function testWithDomainInvalid()
    {
        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage('The provided arg must be of type `string` but `array` given');
        $cookie = new Cookie('cakephp', 'rocks');
        $cookie->withDomain(['oops']);
    }

    /**
     * Test setting domain in cookies
     *
     * @return void
     */
    public function testWithDomain()
    {
        $cookie = new Cookie('cakephp', 'cakephp-rocks');
        $new = $cookie->withDomain('example.com');
        $this->assertNotSame($new, $cookie, 'Should make a clone');
        $this->assertNotContains('example.com', $cookie->toHeaderValue(), 'old instance not modified');
        $this->assertContains('domain=example.com', $new->toHeaderValue());
    }

    /**
     * Test setting path in cookies
     *
     * @return void
     */
    public function testWithPathInvalid()
    {
        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage('The provided arg must be of type `string` but `array` given');
        $cookie = new Cookie('cakephp', 'rocks');
        $cookie->withPath(['oops']);
    }

    /**
     * Test setting path in cookies
     *
     * @return void
     */
    public function testWithPathInvalidConstructor()
    {
        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage('The provided arg must be of type `string` but `integer` given');
        new Cookie('cakephp', 'rocks', null, 123);
    }

    /**
     * Test setting path in cookies
     *
     * @return void
     */
    public function testWithPath()
    {
        $cookie = new Cookie('cakephp', 'cakephp-rocks');
        $new = $cookie->withPath('/api');
        $this->assertNotSame($new, $cookie, 'Should make a clone');
        $this->assertNotContains('path=/api', $cookie->toHeaderValue(), 'old instance not modified');
        $this->assertContains('path=/api', $new->toHeaderValue());
    }

    /**
     * Test default path in cookies
     *
     * @return void
     */
    public function testDefaultPath()
    {
        $cookie = new Cookie('cakephp', 'cakephp-rocks');
        $this->assertContains('path=/', $cookie->toHeaderValue());
    }

    /**
     * Test setting httponly in cookies
     *
     * @return void
     */
    public function testWithHttpOnlyInvalidConstructor()
    {
        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage('The provided arg must be of type `bool` but `string` given');
        new Cookie('cakephp', 'cakephp-rocks', null, '', '', false, 'invalid');
    }

    /**
     * Test setting httponly in cookies
     *
     * @return void
     */
    public function testWithHttpOnlyInvalid()
    {
        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage('The provided arg must be of type `bool` but `string` given');
        $cookie = new Cookie('cakephp', 'cakephp-rocks');
        $cookie->withHttpOnly('no');
    }

    /**
     * Test setting httponly in cookies
     *
     * @return void
     */
    public function testWithHttpOnly()
    {
        $cookie = new Cookie('cakephp', 'cakephp-rocks');
        $new = $cookie->withHttpOnly(true);
        $this->assertNotSame($new, $cookie, 'Should clone');
        $this->assertFalse($cookie->isHttpOnly());
        $this->assertTrue($new->isHttpOnly());
    }

    /**
     * Test setting secure in cookies
     *
     * @return void
     */
    public function testWithSecureInvalidConstructor()
    {
        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage('The provided arg must be of type `bool` but `string` given');
        new Cookie('cakephp', 'cakephp-rocks', null, '', '', 'invalid');
    }

    /**
     * Test setting secure in cookies
     *
     * @return void
     */
    public function testWithSecureInvalid()
    {
        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage('The provided arg must be of type `bool` but `string` given');
        $cookie = new Cookie('cakephp', 'cakephp-rocks');
        $cookie->withSecure('no');
    }

    /**
     * Test setting secure in cookies
     *
     * @return void
     */
    public function testWithSecure()
    {
        $cookie = new Cookie('cakephp', 'cakephp-rocks');
        $new = $cookie->withSecure(true);
        $this->assertNotSame($new, $cookie, 'Should clone');
        $this->assertFalse($cookie->isSecure());
        $this->assertTrue($new->isSecure());
    }

    /**
     * Test the never expiry method
     *
     * @return void
     */
    public function testWithNeverExpire()
    {
        $cookie = new Cookie('cakephp', 'cakephp-rocks');
        $new = $cookie->withNeverExpire();
        $this->assertNotSame($new, $cookie, 'Should clone');
        $this->assertContains('01-Jan-2038', $new->toHeaderValue());
    }

    /**
     * Test the expired method
     *
     * @return void
     */
    public function testWithExpired()
    {
        $cookie = new Cookie('cakephp', 'cakephp-rocks');
        $new = $cookie->withExpired();
        $this->assertNotSame($new, $cookie, 'Should clone');
        $this->assertNotContains('expiry', $cookie->toHeaderValue());

        $this->assertContains('01-Jan-1970', $new->toHeaderValue());
    }

    /**
     * Test the withExpiry method
     *
     * @return void
     */
    public function testWithExpiry()
    {
        $cookie = new Cookie('cakephp', 'cakephp-rocks');
        $new = $cookie->withExpiry(Chronos::createFromDate(2022, 6, 15));
        $this->assertNotSame($new, $cookie, 'Should clone');
        $this->assertNotContains('expires', $cookie->toHeaderValue());

        $this->assertContains('expires=Wed, 15-Jun-2022', $new->toHeaderValue());
    }

    /**
     * Test the withExpiry method changes timezone
     *
     * @return void
     */
    public function testWithExpiryChangesTimezone()
    {
        $cookie = new Cookie('cakephp', 'cakephp-rocks');
        $date = Chronos::createFromDate(2022, 6, 15);
        $date = $date->setTimezone('America/New_York');

        $new = $cookie->withExpiry($date);
        $this->assertNotSame($new, $cookie, 'Should clone');
        $this->assertNotContains('expires', $cookie->toHeaderValue());

        $this->assertContains('expires=Wed, 15-Jun-2022', $new->toHeaderValue());
        $this->assertContains('GMT', $new->toHeaderValue());
        $this->assertSame($date->format('U'), $new->getExpiresTimestamp());
    }

    /**
     * Test the isExpired method
     *
     * @return void
     */
    public function testIsExpired()
    {
        $date = Chronos::now();
        $cookie = new Cookie('cakephp', 'yay');
        $this->assertFalse($cookie->isExpired($date));

        $cookie = new Cookie('cakephp', 'yay', $date);
        $this->assertFalse($cookie->isExpired($date), 'same time, not expired');

        $date = $date->modify('+10 seconds');
        $this->assertTrue($cookie->isExpired($date), 'future now');

        $date = $date->modify('-1 minute');
        $this->assertFalse($cookie->isExpired($date), 'expires later');
    }

    /**
     * Test the withName method
     *
     * @return void
     */
    public function testWithName()
    {
        $cookie = new Cookie('cakephp', 'cakephp-rocks');
        $new = $cookie->withName('user');
        $this->assertNotSame($new, $cookie, 'Should clone');
        $this->assertNotSame('user', $cookie->getName());
        $this->assertSame('user', $new->getName());
    }

    /**
     * Test the withAddedValue method
     *
     * @return void
     */
    public function testWithAddedValue()
    {
        $cookie = new Cookie('cakephp', '{"type":"mvc", "icing": true}');
        $new = $cookie->withAddedValue('type', 'mvc')
            ->withAddedValue('user.name', 'mark');
        $this->assertNotSame($new, $cookie, 'Should clone');
        $this->assertNull($cookie->read('user.name'));
        $this->assertSame('mvc', $new->read('type'));
        $this->assertSame('mark', $new->read('user.name'));
    }

    /**
     * Test the withoutAddedValue method
     *
     * @return void
     */
    public function testWithoutAddedValue()
    {
        $cookie = new Cookie('cakephp', '{"type":"mvc", "user": {"name":"mark"}}');
        $new = $cookie->withoutAddedValue('type', 'mvc')
            ->withoutAddedValue('user.name');
        $this->assertNotSame($new, $cookie, 'Should clone');

        $this->assertNotNull($cookie->read('type'));
        $this->assertNull($new->read('type'));
        $this->assertNull($new->read('user.name'));
    }

    /**
     * Test check() with serialized source data.
     *
     * @return void
     */
    public function testCheckStringSourceData()
    {
        $cookie = new Cookie('cakephp', '{"type":"mvc", "user": {"name":"mark"}}');
        $this->assertTrue($cookie->check('type'));
        $this->assertTrue($cookie->check('user.name'));
        $this->assertFalse($cookie->check('nope'));
        $this->assertFalse($cookie->check('user.nope'));
    }

    /**
     * Test check() with array source data.
     *
     * @return void
     */
    public function testCheckArraySourceData()
    {
        $data = [
            'type' => 'mvc',
            'user' => ['name' => 'mark'],
        ];
        $cookie = new Cookie('cakephp', $data);
        $this->assertTrue($cookie->check('type'));
        $this->assertTrue($cookie->check('user.name'));
        $this->assertFalse($cookie->check('nope'));
        $this->assertFalse($cookie->check('user.nope'));
    }

    /**
     * test read() and set on different types
     *
     * @return void
     */
    public function testReadExpandsOnDemand()
    {
        $data = [
            'username' => 'florian',
            'profile' => [
                'profession' => 'developer',
            ],
        ];
        $cookie = new Cookie('cakephp', json_encode($data));
        $this->assertFalse($cookie->isExpanded());
        $this->assertEquals('developer', $cookie->read('profile.profession'));
        $this->assertTrue($cookie->isExpanded(), 'Cookies expand when read.');

        $cookie = $cookie->withValue(json_encode($data));
        $this->assertTrue($cookie->check('profile.profession'), 'Cookies expand when read.');
        $this->assertTrue($cookie->isExpanded());

        $cookie = $cookie->withValue(json_encode($data))
            ->withAddedValue('face', 'punch');
        $this->assertTrue($cookie->isExpanded());
        $this->assertSame('punch', $cookie->read('face'));
    }

    /**
     * test read() on structured data.
     *
     * @return void
     */
    public function testReadComplexData()
    {
        $data = [
            'username' => 'florian',
            'profile' => [
                'profession' => 'developer',
            ],
        ];
        $cookie = new Cookie('cakephp', $data);

        $result = $cookie->getValue();
        $this->assertEquals($data, $result);

        $result = $cookie->read('foo');
        $this->assertNull($result);

        $result = $cookie->read();
        $this->assertEquals($data, $result);

        $result = $cookie->read('profile.profession');
        $this->assertEquals('developer', $result);
    }

    /**
     * Test reading complex data serialized in 1.x and early 2.x
     *
     * @return void
     */
    public function testReadLegacyComplexData()
    {
        $data = 'key|value,key2|value2';
        $cookie = new Cookie('cakephp', $data);
        $this->assertEquals('value', $cookie->read('key'));
        $this->assertNull($cookie->read('nope'));
    }

    /**
     * Test that toHeaderValue() collapses data.
     *
     * @return void
     */
    public function testToHeaderValueCollapsesComplexData()
    {
        $data = [
            'username' => 'florian',
            'profile' => [
                'profession' => 'developer',
            ],
        ];
        $cookie = new Cookie('cakephp', $data);
        $this->assertEquals('developer', $cookie->read('profile.profession'));

        $expected = '{"username":"florian","profile":{"profession":"developer"}}';
        $this->assertContains(urlencode($expected), $cookie->toHeaderValue());
    }

    /**
     * Tests getting the id
     *
     * @return void
     */
    public function testGetId()
    {
        $cookie = new Cookie('cakephp', 'cakephp-rocks');
        $this->assertEquals('cakephp;;/', $cookie->getId());

        $cookie = new Cookie('CAKEPHP', 'cakephp-rocks');
        $this->assertEquals('CAKEPHP;;/', $cookie->getId());

        $cookie = new Cookie('test', 'val', null, '/path', 'example.com');
        $this->assertEquals('test;example.com;/path', $cookie->getId());
    }
}
